"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.writeLangSmithFeedback = exports.getConnectorName = exports.fetchLangSmithDataset = void 0;
var _langsmith = require("langsmith");
var _langsmith2 = require("@kbn/langchain/server/tracers/langsmith");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Return connector name for the given connectorId/connectors
 *
 * @param connectorId
 * @param connectors
 */
const getConnectorName = (connectorId, connectors) => {
  var _connectors$find;
  return (_connectors$find = connectors.find(c => c.id === connectorId)) === null || _connectors$find === void 0 ? void 0 : _connectors$find.name;
};

/**
 * Fetches a dataset from LangSmith. Note that `client` will use env vars
 *
 * @param datasetName
 * @param logger
 */
exports.getConnectorName = getConnectorName;
const fetchLangSmithDataset = async (datasetName, logger) => {
  if (datasetName === undefined || !(0, _langsmith2.isLangSmithEnabled)()) {
    throw new Error('LangSmith dataset name not provided or LangSmith not enabled');
  }
  try {
    const client = new _langsmith.Client();
    const examples = [];
    for await (const example of client.listExamples({
      datasetName
    })) {
      examples.push(example);
    }

    // Convert to internal Dataset type -- TODO: add generic support for the different LangSmith test dataset formats
    const dataset = examples.map(example => {
      var _ref, _example$outputs;
      return {
        id: example.id,
        input: example.inputs.input,
        reference: (_ref = (_example$outputs = example.outputs) === null || _example$outputs === void 0 ? void 0 : _example$outputs.output) !== null && _ref !== void 0 ? _ref : '',
        tags: [],
        // TODO: Consider adding tags from example data, e.g.: `datasetId:${example.dataset_id}`, `exampleName:${example.name}`
        prediction: undefined
      };
    });
    return dataset;
  } catch (e) {
    logger.error(`Error fetching dataset from LangSmith: ${e.message}`);
    return [];
  }
};

/**
 * Write Feedback to LangSmith for a given Run
 *
 * @param run
 * @param evaluationId
 * @param logger
 */
exports.fetchLangSmithDataset = fetchLangSmithDataset;
const writeLangSmithFeedback = async (run, evaluationId, logger) => {
  try {
    var _run$feedback_stats, _run$feedback_stats2, _run$feedback_stats3, _run$feedback_stats4, _run$feedback_stats5, _run$feedback_stats6, _run$feedback_stats7, _run$feedback_stats8, _run$feedback_stats9;
    const client = new _langsmith.Client();
    const feedback = {
      score: (_run$feedback_stats = run.feedback_stats) === null || _run$feedback_stats === void 0 ? void 0 : _run$feedback_stats.score,
      value: (_run$feedback_stats2 = run.feedback_stats) === null || _run$feedback_stats2 === void 0 ? void 0 : _run$feedback_stats2.value,
      correction: (_run$feedback_stats3 = run.feedback_stats) === null || _run$feedback_stats3 === void 0 ? void 0 : _run$feedback_stats3.correction,
      comment: (_run$feedback_stats4 = run.feedback_stats) === null || _run$feedback_stats4 === void 0 ? void 0 : _run$feedback_stats4.comment,
      sourceInfo: (_run$feedback_stats5 = run.feedback_stats) === null || _run$feedback_stats5 === void 0 ? void 0 : _run$feedback_stats5.sourceInfo,
      feedbackSourceType: (_run$feedback_stats6 = run.feedback_stats) === null || _run$feedback_stats6 === void 0 ? void 0 : _run$feedback_stats6.feedbackSourceType,
      sourceRunId: (_run$feedback_stats7 = run.feedback_stats) === null || _run$feedback_stats7 === void 0 ? void 0 : _run$feedback_stats7.sourceRunId,
      feedbackId: (_run$feedback_stats8 = run.feedback_stats) === null || _run$feedback_stats8 === void 0 ? void 0 : _run$feedback_stats8.feedbackId,
      eager: (_run$feedback_stats9 = run.feedback_stats) === null || _run$feedback_stats9 === void 0 ? void 0 : _run$feedback_stats9.eager
    };
    await client.createFeedback(run.id, evaluationId, feedback);
    const runUrl = await client.getRunUrl({
      run
    });
    return runUrl;
  } catch (e) {
    logger.error(`Error writing feedback to LangSmith: ${e.message}`);
    return '';
  }
};
exports.writeLangSmithFeedback = writeLangSmithFeedback;