"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postEvaluateRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _uuid = require("uuid");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _server = require("@kbn/langchain/server");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _constants = require("../knowledge_base/constants");
var _build_response = require("../../lib/build_response");
var _constants2 = require("../../../common/constants");
var _evaluation = require("../../lib/model_evaluator/evaluation");
var _utils = require("../../lib/model_evaluator/output_index/utils");
var _utils2 = require("./utils");
var _helpers = require("../helpers");
var _executors = require("../../lib/langchain/executors");
var _elasticsearch_store = require("../../lib/langchain/elasticsearch_store/elasticsearch_store");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * To support additional Agent Executors from the UI, add them to this map
 * and reference your specific AgentExecutor function
 */

const DEFAULT_SIZE = 20;
const postEvaluateRoute = (router, getElser) => {
  router.versioned.post({
    access: _elasticAssistantCommon.INTERNAL_API_ACCESS,
    path: _constants2.EVALUATE,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.PostEvaluateBody),
        query: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.PostEvaluateRequestQuery)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.PostEvaluateResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;

    // Validate evaluation feature is enabled
    const pluginName = (0, _helpers.getPluginNameFromRequest)({
      request,
      defaultPluginName: _helpers.DEFAULT_PLUGIN_NAME,
      logger
    });
    const registeredFeatures = assistantContext.getRegisteredFeatures(pluginName);
    if (!registeredFeatures.assistantModelEvaluation) {
      return response.notFound();
    }
    try {
      var _request$query$models, _request$query$agents, _await$assistantConte;
      const evaluationId = (0, _uuid.v4)();
      const {
        evalModel,
        evaluationType,
        outputIndex,
        datasetName,
        projectName = 'default',
        runName = evaluationId
      } = request.query;
      const {
        dataset: customDataset = [],
        evalPrompt
      } = request.body;
      const connectorIds = ((_request$query$models = request.query.models) === null || _request$query$models === void 0 ? void 0 : _request$query$models.split(',')) || [];
      const agentNames = ((_request$query$agents = request.query.agents) === null || _request$query$agents === void 0 ? void 0 : _request$query$agents.split(',')) || [];
      const dataset = datasetName != null ? await (0, _utils2.fetchLangSmithDataset)(datasetName, logger) : customDataset;
      logger.info('postEvaluateRoute:');
      logger.info(`request.query:\n${JSON.stringify(request.query, null, 2)}`);
      logger.info(`request.body:\n${JSON.stringify(request.body, null, 2)}`);
      logger.info(`Evaluation ID: ${evaluationId}`);
      const totalExecutions = connectorIds.length * agentNames.length * dataset.length;
      logger.info('Creating agents:');
      logger.info(`\tconnectors/models: ${connectorIds.length}`);
      logger.info(`\tagents: ${agentNames.length}`);
      logger.info(`\tdataset: ${dataset.length}`);
      logger.warn(`\ttotal baseline agent executions: ${totalExecutions} `);
      if (totalExecutions > 50) {
        logger.warn(`Total baseline agent executions >= 50! This may take a while, and cost some money...`);
      }

      // Get the actions plugin start contract from the request context for the agents
      const actions = (await context.elasticAssistant).actions;

      // Fetch all connectors from the actions plugin, so we can set the appropriate `llmType` on ActionsClientLlm
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const connectors = await actionsClient.getBulk({
        ids: connectorIds,
        throwIfSystemAction: false
      });

      // Fetch any tools registered by the request's originating plugin
      const assistantTools = (await context.elasticAssistant).getRegisteredTools(pluginName);

      // Get a scoped esClient for passing to the agents for retrieval, and
      // writing results to the output index
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // Default ELSER model
      const elserId = await getElser();

      // Skeleton request from route to pass to the agents
      // params will be passed to the actions executor
      const skeletonRequest = {
        ...request,
        body: {
          alertsIndexPattern: '',
          allow: [],
          allowReplacement: [],
          subAction: 'invokeAI',
          // The actionTypeId is irrelevant when used with the invokeAI subaction
          actionTypeId: '.gen-ai',
          replacements: {},
          size: DEFAULT_SIZE,
          conversationId: ''
        }
      };

      // Create an ElasticsearchStore for KB interactions
      // Setup with kbDataClient if `enableKnowledgeBaseByDefault` FF is enabled
      const enableKnowledgeBaseByDefault = assistantContext.getRegisteredFeatures(pluginName).assistantKnowledgeBaseByDefault;
      const bedrockChatEnabled = assistantContext.getRegisteredFeatures(pluginName).assistantBedrockChat;
      const kbDataClient = enableKnowledgeBaseByDefault ? (_await$assistantConte = await assistantContext.getAIAssistantKnowledgeBaseDataClient()) !== null && _await$assistantConte !== void 0 ? _await$assistantConte : undefined : undefined;
      const kbIndex = enableKnowledgeBaseByDefault && kbDataClient != null ? kbDataClient.indexTemplateAndPattern.alias : _constants.KNOWLEDGE_BASE_INDEX_PATTERN;
      const esStore = new _elasticsearch_store.ElasticsearchStore(esClient, kbIndex, logger, telemetry, elserId, _constants.ESQL_RESOURCE, kbDataClient);

      // Create an array of executor functions to call in batches
      // One for each connector/model + agent combination
      // Hoist `langChainMessages` so they can be batched by dataset.input in the evaluator
      const agents = [];
      connectorIds.forEach(connectorId => {
        agentNames.forEach(agentName => {
          var _getConnectorName;
          logger.info(`Creating agent: ${connectorId} + ${agentName}`);
          const connectorName = (_getConnectorName = (0, _utils2.getConnectorName)(connectorId, connectors)) !== null && _getConnectorName !== void 0 ? _getConnectorName : '[unknown connector]';
          const detailedRunName = `${runName} - ${connectorName} + ${agentName}`;
          agents.push({
            agentEvaluator: async (langChainMessages, exampleId) => {
              const evalResult = await _executors.AGENT_EXECUTOR_MAP[agentName]({
                actionsClient,
                assistantTools,
                bedrockChatEnabled,
                connectorId,
                esClient,
                esStore,
                isStream: false,
                langChainMessages,
                llmType: 'openai',
                logger,
                request: skeletonRequest,
                traceOptions: {
                  exampleId,
                  projectName,
                  runName: detailedRunName,
                  evaluationId,
                  tags: ['security-assistant-prediction', ...(connectorName != null ? [connectorName] : []), runName],
                  tracers: (0, _langsmith.getLangSmithTracer)({
                    projectName: detailedRunName,
                    exampleId,
                    logger
                  })
                },
                replacements: {}
              });
              return evalResult.body;
            },
            metadata: {
              connectorName,
              runName: detailedRunName
            }
          });
        });
      });
      logger.info(`Agents created: ${agents.length}`);

      // Evaluator Model is optional to support just running predictions
      const evaluatorModel = evalModel == null || evalModel === '' ? undefined : new _server.ActionsClientLlm({
        actionsClient,
        connectorId: evalModel,
        logger,
        model: skeletonRequest.body.model
      });
      const {
        evaluationResults,
        evaluationSummary
      } = await (0, _evaluation.performEvaluation)({
        agentExecutorEvaluators: agents,
        dataset,
        evaluationId,
        evaluatorModel,
        evaluationPrompt: evalPrompt,
        evaluationType,
        logger,
        runName
      });
      logger.info(`Writing evaluation results to index: ${outputIndex}`);
      await (0, _utils.setupEvaluationIndex)({
        esClient,
        index: outputIndex,
        logger
      });
      await (0, _utils.indexEvaluations)({
        esClient,
        evaluationResults,
        evaluationSummary,
        index: outputIndex,
        logger
      });
      return response.ok({
        body: {
          evaluationId,
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      const resp = (0, _build_response.buildResponse)(response);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postEvaluateRoute = postEvaluateRoute;