"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAttackDiscoveryStatusToRunning = exports.updateAttackDiscoveryStatusToCanceled = exports.updateAttackDiscoveryLastViewedAt = exports.updateAttackDiscoveries = exports.handleToolError = exports.getAttackDiscoveryStats = exports.getAssistantToolParams = exports.getAssistantTool = exports.attackDiscoveryStatus = exports.addGenerationInterval = exports.REQUIRED_FOR_ATTACK_DISCOVERY = void 0;
var _uuid = require("uuid");
var _server = require("@kbn/langchain/server");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _moment = _interopRequireDefault(require("moment/moment"));
var _fp = require("lodash/fp");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _utils = require("../utils");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REQUIRED_FOR_ATTACK_DISCOVERY = exports.REQUIRED_FOR_ATTACK_DISCOVERY = [{
  id: (0, _uuid.v4)(),
  field: '_id',
  allowed: true,
  anonymized: true
}, {
  id: (0, _uuid.v4)(),
  field: 'kibana.alert.original_time',
  allowed: true,
  anonymized: false
}];
const getAssistantToolParams = ({
  actionsClient,
  alertsIndexPattern,
  anonymizationFields,
  apiConfig,
  esClient,
  connectorTimeout,
  langChainTimeout,
  langSmithProject,
  langSmithApiKey,
  logger,
  latestReplacements,
  onNewReplacements,
  request,
  size
}) => {
  const traceOptions = {
    projectName: langSmithProject,
    tracers: [...(0, _langsmith.getLangSmithTracer)({
      apiKey: langSmithApiKey,
      projectName: langSmithProject,
      logger
    })]
  };
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId: apiConfig.connectorId,
    llmType: (0, _utils.getLlmType)(apiConfig.actionTypeId),
    logger,
    temperature: 0,
    // zero temperature for attack discovery, because we want structured JSON output
    timeout: connectorTimeout,
    traceOptions
  });
  return formatAssistantToolParams({
    alertsIndexPattern,
    anonymizationFields,
    esClient,
    latestReplacements,
    langChainTimeout,
    llm,
    logger,
    onNewReplacements,
    request,
    size
  });
};
exports.getAssistantToolParams = getAssistantToolParams;
const formatAssistantToolParams = ({
  alertsIndexPattern,
  anonymizationFields,
  esClient,
  langChainTimeout,
  latestReplacements,
  llm,
  logger,
  onNewReplacements,
  request,
  size
}) => ({
  alertsIndexPattern,
  anonymizationFields: [...(anonymizationFields !== null && anonymizationFields !== void 0 ? anonymizationFields : []), ...REQUIRED_FOR_ATTACK_DISCOVERY],
  isEnabledKnowledgeBase: false,
  // not required for attack discovery
  chain: undefined,
  // not required for attack discovery
  esClient,
  langChainTimeout,
  llm,
  logger,
  modelExists: false,
  // not required for attack discovery
  onNewReplacements,
  replacements: latestReplacements,
  request,
  size
});
const attackDiscoveryStatus = exports.attackDiscoveryStatus = {
  canceled: 'canceled',
  failed: 'failed',
  running: 'running',
  succeeded: 'succeeded'
};
const MAX_GENERATION_INTERVALS = 5;
const addGenerationInterval = (generationIntervals, generationInterval) => {
  const newGenerationIntervals = [generationInterval, ...generationIntervals];
  if (newGenerationIntervals.length > MAX_GENERATION_INTERVALS) {
    return newGenerationIntervals.slice(0, MAX_GENERATION_INTERVALS); // Return the first MAX_GENERATION_INTERVALS items
  }
  return newGenerationIntervals;
};
exports.addGenerationInterval = addGenerationInterval;
const updateAttackDiscoveryStatusToRunning = async (dataClient, authenticatedUser, apiConfig) => {
  const foundAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findAttackDiscoveryByConnectorId({
    connectorId: apiConfig.connectorId,
    authenticatedUser
  }));
  const currentAd = foundAttackDiscovery ? await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      backingIndex: foundAttackDiscovery.backingIndex,
      id: foundAttackDiscovery.id,
      status: attackDiscoveryStatus.running
    },
    authenticatedUser
  })) : await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createAttackDiscovery({
    attackDiscoveryCreate: {
      apiConfig,
      attackDiscoveries: [],
      status: attackDiscoveryStatus.running
    },
    authenticatedUser
  }));
  if (!currentAd) {
    throw new Error(`Could not ${foundAttackDiscovery ? 'update' : 'create'} attack discovery for connectorId: ${apiConfig.connectorId}`);
  }
  return {
    attackDiscoveryId: currentAd.id,
    currentAd
  };
};
exports.updateAttackDiscoveryStatusToRunning = updateAttackDiscoveryStatusToRunning;
const updateAttackDiscoveryStatusToCanceled = async (dataClient, authenticatedUser, connectorId) => {
  const foundAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findAttackDiscoveryByConnectorId({
    connectorId,
    authenticatedUser
  }));
  if (foundAttackDiscovery == null) {
    throw new Error(`Could not find attack discovery for connector id: ${connectorId}`);
  }
  if (foundAttackDiscovery.status !== 'running') {
    throw new Error(`Connector id ${connectorId} does not have a running attack discovery, and therefore cannot be canceled.`);
  }
  const updatedAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      backingIndex: foundAttackDiscovery.backingIndex,
      id: foundAttackDiscovery.id,
      status: attackDiscoveryStatus.canceled
    },
    authenticatedUser
  }));
  if (!updatedAttackDiscovery) {
    throw new Error(`Could not update attack discovery for connector id: ${connectorId}`);
  }
  return updatedAttackDiscovery;
};
exports.updateAttackDiscoveryStatusToCanceled = updateAttackDiscoveryStatusToCanceled;
const getDataFromJSON = adStringified => {
  const {
    alertsContextCount,
    attackDiscoveries
  } = JSON.parse(adStringified);
  return {
    alertsContextCount,
    attackDiscoveries
  };
};
const updateAttackDiscoveries = async ({
  apiConfig,
  attackDiscoveryId,
  authenticatedUser,
  dataClient,
  latestReplacements,
  logger,
  rawAttackDiscoveries,
  size,
  startTime,
  telemetry
}) => {
  try {
    if (rawAttackDiscoveries == null) {
      throw new Error('tool returned no attack discoveries');
    }
    const currentAd = await dataClient.getAttackDiscovery({
      id: attackDiscoveryId,
      authenticatedUser
    });
    if (currentAd === null || (currentAd === null || currentAd === void 0 ? void 0 : currentAd.status) === 'canceled') {
      return;
    }
    const endTime = (0, _moment.default)();
    const durationMs = endTime.diff(startTime);
    const {
      alertsContextCount,
      attackDiscoveries
    } = getDataFromJSON(rawAttackDiscoveries);
    const updateProps = {
      alertsContextCount,
      attackDiscoveries,
      status: attackDiscoveryStatus.succeeded,
      ...(alertsContextCount === 0 || attackDiscoveries === 0 ? {} : {
        generationIntervals: addGenerationInterval(currentAd.generationIntervals, {
          durationMs,
          date: new Date().toISOString()
        })
      }),
      id: attackDiscoveryId,
      replacements: latestReplacements,
      backingIndex: currentAd.backingIndex
    };
    await dataClient.updateAttackDiscovery({
      attackDiscoveryUpdateProps: updateProps,
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.ATTACK_DISCOVERY_SUCCESS_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      alertsContextCount: updateProps.alertsContextCount,
      alertsCount: (0, _fp.uniq)(updateProps.attackDiscoveries.flatMap(attackDiscovery => attackDiscovery.alertIds)).length,
      configuredAlertsCount: size,
      discoveriesGenerated: updateProps.attackDiscoveries.length,
      durationMs,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    logger.error(updateErr);
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.ATTACK_DISCOVERY_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
};
exports.updateAttackDiscoveries = updateAttackDiscoveries;
const handleToolError = async ({
  apiConfig,
  attackDiscoveryId,
  authenticatedUser,
  dataClient,
  err,
  latestReplacements,
  logger,
  telemetry
}) => {
  try {
    logger.error(err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    const currentAd = await dataClient.getAttackDiscovery({
      id: attackDiscoveryId,
      authenticatedUser
    });
    if (currentAd === null || (currentAd === null || currentAd === void 0 ? void 0 : currentAd.status) === 'canceled') {
      return;
    }
    await dataClient.updateAttackDiscovery({
      attackDiscoveryUpdateProps: {
        attackDiscoveries: [],
        status: attackDiscoveryStatus.failed,
        id: attackDiscoveryId,
        replacements: latestReplacements,
        backingIndex: currentAd.backingIndex,
        failureReason: error.message
      },
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.ATTACK_DISCOVERY_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: error.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.ATTACK_DISCOVERY_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
};
exports.handleToolError = handleToolError;
const getAssistantTool = (getRegisteredTools, pluginName) => {
  // get the attack discovery tool:
  const assistantTools = getRegisteredTools(pluginName);
  return assistantTools.find(tool => tool.id === 'attack-discovery');
};
exports.getAssistantTool = getAssistantTool;
const updateAttackDiscoveryLastViewedAt = async ({
  connectorId,
  authenticatedUser,
  dataClient
}) => {
  const attackDiscovery = await dataClient.findAttackDiscoveryByConnectorId({
    connectorId,
    authenticatedUser
  });
  if (attackDiscovery == null) {
    return null;
  }

  // update lastViewedAt time as this is the function used for polling by connectorId
  return dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      id: attackDiscovery.id,
      lastViewedAt: new Date().toISOString(),
      backingIndex: attackDiscovery.backingIndex
    },
    authenticatedUser
  });
};
exports.updateAttackDiscoveryLastViewedAt = updateAttackDiscoveryLastViewedAt;
const getAttackDiscoveryStats = async ({
  authenticatedUser,
  dataClient
}) => {
  const attackDiscoveries = await dataClient.findAllAttackDiscoveries({
    authenticatedUser
  });
  return attackDiscoveries.map(ad => {
    const updatedAt = (0, _moment.default)(ad.updatedAt);
    const lastViewedAt = (0, _moment.default)(ad.lastViewedAt);
    const timeSinceLastViewed = updatedAt.diff(lastViewedAt);
    const hasViewed = timeSinceLastViewed <= 0;
    const discoveryCount = ad.attackDiscoveries.length;
    return {
      hasViewed,
      status: ad.status,
      count: discoveryCount,
      connectorId: ad.apiConfig.connectorId
    };
  });
};
exports.getAttackDiscoveryStats = getAttackDiscoveryStats;