"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callAssistantGraph = void 0;
var _chains = require("langchain/chains");
var _server = require("@kbn/langchain/server");
var _agents = require("langchain/agents");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _utils = require("../../../../routes/utils");
var _prompts = require("./prompts");
var _graph = require("./graph");
var _helpers = require("./helpers");
var _helpers2 = require("../../../../ai_assistant_data_clients/anonymization_fields/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Drop in replacement for the existing `callAgentExecutor` that uses LangGraph
 */
const callAssistantGraph = async ({
  abortSignal,
  actionsClient,
  alertsIndexPattern,
  assistantTools = [],
  bedrockChatEnabled,
  connectorId,
  conversationId,
  dataClients,
  esClient,
  esStore,
  langChainMessages,
  llmType,
  logger: parentLogger,
  isStream = false,
  onLlmResponse,
  onNewReplacements,
  replacements,
  request,
  size,
  traceOptions,
  responseLanguage = 'English'
}) => {
  var _dataClients$anonymiz, _await$dataClients$kb, _dataClients$kbDataCl, _traceOptions$project, _latestMessage$;
  const logger = parentLogger.get('defaultAssistantGraph');
  const isOpenAI = llmType === 'openai';
  const llmClass = (0, _utils.getLlmClass)(llmType, bedrockChatEnabled);
  const getLlmInstance = () => new llmClass({
    actionsClient,
    connectorId,
    llmType,
    logger,
    // possible client model override,
    // let this be undefined otherwise so the connector handles the model
    model: request.body.model,
    // ensure this is defined because we default to it in the language_models
    // This is where the LangSmith logs (Metadata > Invocation Params) are set
    temperature: (0, _server.getDefaultArguments)(llmType).temperature,
    signal: abortSignal,
    streaming: isStream,
    // prevents the agent from retrying on failure
    // failure could be due to bad connector, we should deliver that result to the client asap
    maxRetries: 0
  });
  const llm = getLlmInstance();
  const anonymizationFieldsRes = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$anonymiz = dataClients.anonymizationFieldsDataClient) === null || _dataClients$anonymiz === void 0 ? void 0 : _dataClients$anonymiz.findDocuments({
    perPage: 1000,
    page: 1
  }));
  const anonymizationFields = anonymizationFieldsRes ? (0, _helpers2.transformESSearchToAnonymizationFields)(anonymizationFieldsRes.data) : undefined;
  const latestMessage = langChainMessages.slice(-1); // the last message

  const modelExists = await esStore.isModelInstalled();

  // Create a chain that uses the ELSER backed ElasticsearchStore, override k=10 for esql query generation for now
  const chain = _chains.RetrievalQAChain.fromLLM(getLlmInstance(), esStore.asRetriever(10));

  // Check if KB is available
  const isEnabledKnowledgeBase = (_await$dataClients$kb = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$kbDataCl = dataClients.kbDataClient) === null || _dataClients$kbDataCl === void 0 ? void 0 : _dataClients$kbDataCl.isModelDeployed())) !== null && _await$dataClients$kb !== void 0 ? _await$dataClients$kb : false;

  // Fetch any applicable tools that the source plugin may have registered
  const assistantToolParams = {
    alertsIndexPattern,
    anonymizationFields,
    chain,
    esClient,
    isEnabledKnowledgeBase,
    kbDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.kbDataClient,
    llm,
    logger,
    modelExists,
    onNewReplacements,
    replacements,
    request,
    size
  };
  const tools = assistantTools.flatMap(tool => {
    var _tool$getTool;
    return (_tool$getTool = tool.getTool(assistantToolParams)) !== null && _tool$getTool !== void 0 ? _tool$getTool : [];
  });
  const agentRunnable = isOpenAI ? await (0, _agents.createOpenAIFunctionsAgent)({
    llm,
    tools,
    prompt: _prompts.openAIFunctionAgentPrompt,
    streamRunnable: isStream
  }) : llmType && ['bedrock', 'gemini'].includes(llmType) && bedrockChatEnabled ? await (0, _agents.createToolCallingAgent)({
    llm,
    tools,
    prompt: llmType === 'bedrock' ? _prompts.bedrockToolCallingAgentPrompt : _prompts.geminiToolCallingAgentPrompt,
    streamRunnable: isStream
  }) : await (0, _agents.createStructuredChatAgent)({
    llm,
    tools,
    prompt: _prompts.structuredChatAgentPrompt,
    streamRunnable: isStream
  });
  const apmTracer = new _apm.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  const assistantGraph = (0, _graph.getDefaultAssistantGraph)({
    agentRunnable,
    conversationId,
    dataClients,
    llm,
    // we need to pass it like this or streaming does not work for bedrock
    getLlmInstance,
    logger,
    tools,
    responseLanguage,
    replacements,
    llmType,
    bedrockChatEnabled,
    isStreaming: isStream
  });
  const inputs = {
    input: (_latestMessage$ = latestMessage[0]) === null || _latestMessage$ === void 0 ? void 0 : _latestMessage$.content
  };
  if (isStream) {
    return (0, _helpers.streamGraph)({
      apmTracer,
      assistantGraph,
      llmType,
      bedrockChatEnabled,
      inputs,
      logger,
      onLlmResponse,
      request,
      traceOptions
    });
  }
  const graphResponse = await (0, _helpers.invokeGraph)({
    apmTracer,
    assistantGraph,
    inputs,
    onLlmResponse,
    traceOptions
  });
  return {
    body: {
      connector_id: connectorId,
      data: graphResponse.output,
      trace_data: graphResponse.traceData,
      replacements,
      status: 'ok',
      ...(graphResponse.conversationId ? {
        conversationId: graphResponse.conversationId
      } : {})
    },
    headers: {
      'content-type': 'application/json'
    }
  };
};
exports.callAssistantGraph = callAssistantGraph;