"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefaultAssistantGraph = exports.DEFAULT_ASSISTANT_GRAPH_ID = void 0;
var _langgraph = require("@langchain/langgraph");
var _should_continue = require("./nodes/should_continue");
var _run_agent = require("./nodes/run_agent");
var _execute_tools = require("./nodes/execute_tools");
var _generate_chat_title = require("./nodes/generate_chat_title");
var _get_persisted_conversation = require("./nodes/get_persisted_conversation");
var _persist_conversation_changes = require("./nodes/persist_conversation_changes");
var _respond = require("./nodes/respond");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_ASSISTANT_GRAPH_ID = exports.DEFAULT_ASSISTANT_GRAPH_ID = 'Default Security Assistant Graph';
/**
 * Returns a compiled default assistant graph
 */
const getDefaultAssistantGraph = ({
  agentRunnable,
  conversationId,
  dataClients,
  getLlmInstance,
  llm,
  logger,
  responseLanguage,
  tools,
  replacements,
  llmType,
  bedrockChatEnabled,
  isStreaming
}) => {
  try {
    // Default graph state
    const graphState = {
      input: {
        value: (x, y) => y !== null && y !== void 0 ? y : x,
        default: () => ''
      },
      steps: {
        value: (x, y) => x.concat(y),
        default: () => []
      },
      agentOutcome: {
        value: (x, y) => y !== null && y !== void 0 ? y : x,
        default: () => undefined
      },
      messages: {
        value: (x, y) => y !== null && y !== void 0 ? y : x,
        default: () => []
      },
      chatTitle: {
        value: (x, y) => y !== null && y !== void 0 ? y : x,
        default: () => ''
      },
      conversation: {
        value: (x, y) => y !== null && y !== void 0 ? y : x,
        default: () => undefined
      }
    };

    // Default node parameters
    const nodeParams = {
      model: llm,
      logger
    };

    // Create nodes
    const runAgentNode = (state, config) => (0, _run_agent.runAgent)({
      ...nodeParams,
      agentRunnable,
      config,
      dataClients,
      logger: logger.get(_run_agent.AGENT_NODE),
      state
    });
    const executeToolsNode = (state, config) => (0, _execute_tools.executeTools)({
      ...nodeParams,
      config,
      logger: logger.get(_execute_tools.TOOLS_NODE),
      state,
      tools
    });
    const generateChatTitleNode = state => (0, _generate_chat_title.generateChatTitle)({
      ...nodeParams,
      state,
      responseLanguage
    });
    const getPersistedConversationNode = state => (0, _get_persisted_conversation.getPersistedConversation)({
      ...nodeParams,
      state,
      conversationsDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.conversationsDataClient,
      conversationId
    });
    const persistConversationChangesNode = state => (0, _persist_conversation_changes.persistConversationChanges)({
      ...nodeParams,
      state,
      conversationsDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.conversationsDataClient,
      conversationId,
      replacements
    });
    const respondNode = state => (0, _respond.respond)({
      ...nodeParams,
      llm: getLlmInstance(),
      state
    });
    const shouldContinueEdge = state => (0, _should_continue.shouldContinue)({
      ...nodeParams,
      state
    });
    const shouldContinueGenerateTitleEdge = state => (0, _should_continue.shouldContinueGenerateTitle)({
      ...nodeParams,
      state
    });
    const shouldContinueGetConversationEdge = state => (0, _should_continue.shouldContinueGetConversation)({
      ...nodeParams,
      state,
      conversationId
    });

    // Put together a new graph using the nodes and default state from above
    const graph = new _langgraph.StateGraph({
      channels: graphState
    });
    // Define the nodes to cycle between
    graph.addNode(_get_persisted_conversation.GET_PERSISTED_CONVERSATION_NODE, getPersistedConversationNode);
    graph.addNode(_generate_chat_title.GENERATE_CHAT_TITLE_NODE, generateChatTitleNode);
    graph.addNode(_persist_conversation_changes.PERSIST_CONVERSATION_CHANGES_NODE, persistConversationChangesNode);
    graph.addNode(_run_agent.AGENT_NODE, runAgentNode);
    graph.addNode(_execute_tools.TOOLS_NODE, executeToolsNode);
    const hasRespondStep = isStreaming && bedrockChatEnabled && llmType === 'bedrock';
    if (hasRespondStep) {
      graph.addNode(_respond.RESPOND_NODE, respondNode);
      graph.addEdge(_respond.RESPOND_NODE, _langgraph.END);
    }

    // Add edges, alternating between agent and action until finished
    graph.addConditionalEdges(_langgraph.START, shouldContinueGetConversationEdge, {
      continue: _get_persisted_conversation.GET_PERSISTED_CONVERSATION_NODE,
      end: _run_agent.AGENT_NODE
    });
    graph.addConditionalEdges(_get_persisted_conversation.GET_PERSISTED_CONVERSATION_NODE, shouldContinueGenerateTitleEdge, {
      continue: _generate_chat_title.GENERATE_CHAT_TITLE_NODE,
      end: _persist_conversation_changes.PERSIST_CONVERSATION_CHANGES_NODE
    });
    graph.addEdge(_generate_chat_title.GENERATE_CHAT_TITLE_NODE, _persist_conversation_changes.PERSIST_CONVERSATION_CHANGES_NODE);
    graph.addEdge(_persist_conversation_changes.PERSIST_CONVERSATION_CHANGES_NODE, _run_agent.AGENT_NODE);
    // Add conditional edge for basic routing
    graph.addConditionalEdges(_run_agent.AGENT_NODE, shouldContinueEdge, {
      continue: _execute_tools.TOOLS_NODE,
      end: hasRespondStep ? _respond.RESPOND_NODE : _langgraph.END
    });
    graph.addEdge(_execute_tools.TOOLS_NODE, _run_agent.AGENT_NODE);
    // Compile the graph
    return graph.compile();
  } catch (e) {
    throw new Error(`Unable to compile DefaultAssistantGraph\n${e}`);
  }
};
exports.getDefaultAssistantGraph = getDefaultAssistantGraph;