"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callOpenAIFunctionsExecutor = exports.OPEN_AI_FUNCTIONS_AGENT_EXECUTOR_ID = void 0;
var _agents = require("langchain/agents");
var _chains = require("langchain/chains");
var _memory = require("langchain/memory");
var _chain = require("langchain/tools/chain");
var _server = require("@kbn/langchain/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _with_assistant_span = require("../tracers/apm/with_assistant_span");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OPEN_AI_FUNCTIONS_AGENT_EXECUTOR_ID = exports.OPEN_AI_FUNCTIONS_AGENT_EXECUTOR_ID = 'Elastic AI Assistant Agent Executor (OpenAI Functions)';

/**
 * This is an agent executor to be used with the model evaluation API for benchmarking.
 * Currently just a copy of `callAgentExecutor`, but using the `openai-functions` agent type.
 *
 * NOTE: This is not to be used in production as-is, and must be used with an OpenAI ConnectorId
 */
const callOpenAIFunctionsExecutor = async ({
  actionsClient,
  connectorId,
  esClient,
  esStore,
  langChainMessages,
  llmType,
  logger,
  request,
  traceOptions
}) => {
  var _traceOptions$project;
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId,
    llmType,
    logger,
    model: request.body.model
  });
  const pastMessages = langChainMessages.slice(0, -1); // all but the last message
  const latestMessage = langChainMessages.slice(-1); // the last message

  const memory = new _memory.BufferMemory({
    chatHistory: new _memory.ChatMessageHistory(pastMessages),
    memoryKey: 'chat_history',
    // this is the key expected by https://github.com/langchain-ai/langchainjs/blob/a13a8969345b0f149c1ca4a120d63508b06c52a5/langchain/src/agents/initialize.ts#L166
    inputKey: 'input',
    outputKey: 'output',
    returnMessages: true
  });
  const modelExists = await esStore.isModelInstalled();
  if (!modelExists) {
    throw new Error('Please ensure ELSER is configured to use the Knowledge Base, otherwise disable the Knowledge Base in Advanced Settings to continue.');
  }

  // Create a chain that uses the ELSER backed ElasticsearchStore, override k=10 for esql query generation for now
  const chain = _chains.RetrievalQAChain.fromLLM(llm, esStore.asRetriever(10));

  // TODO: Dependency inject these tools
  const tools = [new _chain.ChainTool({
    name: 'ESQLKnowledgeBaseTool',
    description: 'Call this for knowledge on how to build an ESQL query, or answer questions about the ES|QL query language.',
    chain,
    tags: ['esql', 'query-generation', 'knowledge-base']
  })];
  const executor = await (0, _agents.initializeAgentExecutorWithOptions)(tools, llm, {
    agentType: 'openai-functions',
    memory,
    verbose: false
  });

  // Sets up tracer for tracing executions to APM. See x-pack/plugins/elastic_assistant/server/lib/langchain/tracers/README.mdx
  // If LangSmith env vars are set, executions will be traced there as well. See https://docs.smith.langchain.com/tracing
  const apmTracer = new _apm.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  let traceData;

  // Wrap executor call with an APM span for instrumentation
  const langChainResponse = await (0, _with_assistant_span.withAssistantSpan)(OPEN_AI_FUNCTIONS_AGENT_EXECUTOR_ID, async span => {
    var _span$transaction, _traceOptions$tracers, _traceOptions$tags;
    if ((span === null || span === void 0 ? void 0 : (_span$transaction = span.transaction) === null || _span$transaction === void 0 ? void 0 : _span$transaction.ids['transaction.id']) != null && (span === null || span === void 0 ? void 0 : span.ids['trace.id']) != null) {
      traceData = {
        // Transactions ID since this span is the parent
        transaction_id: span.transaction.ids['transaction.id'],
        trace_id: span.ids['trace.id']
      };
      span.addLabels({
        evaluationId: traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.evaluationId
      });
    }
    return executor.call({
      input: latestMessage[0].content
    }, {
      callbacks: [apmTracer, ...((_traceOptions$tracers = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers !== void 0 ? _traceOptions$tracers : [])],
      runName: OPEN_AI_FUNCTIONS_AGENT_EXECUTOR_ID,
      tags: (_traceOptions$tags = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tags) !== null && _traceOptions$tags !== void 0 ? _traceOptions$tags : []
    });
  });
  return {
    body: {
      connector_id: connectorId,
      data: langChainResponse.output,
      // the response from the actions framework
      trace_data: traceData,
      status: 'ok'
    },
    headers: {
      'content-type': 'application/json'
    },
    connector_id: connectorId,
    data: langChainResponse.output,
    // the response from the actions framework
    trace_data: traceData,
    status: 'ok'
  };
};
exports.callOpenAIFunctionsExecutor = callOpenAIFunctionsExecutor;