"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callAgentExecutor = exports.DEFAULT_AGENT_EXECUTOR_ID = void 0;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _agents = require("langchain/agents");
var _memory = require("langchain/memory");
var _server = require("@kbn/ml-response-stream/server");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _chains = require("langchain/chains");
var _server2 = require("@kbn/langchain/server");
var _prompts = require("@langchain/core/prompts");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _with_assistant_span = require("../tracers/apm/with_assistant_span");
var _utils = require("../../../routes/utils");
var _helpers = require("../../../ai_assistant_data_clients/anonymization_fields/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_AGENT_EXECUTOR_ID = exports.DEFAULT_AGENT_EXECUTOR_ID = 'Elastic AI Assistant Agent Executor';

/**
 * The default agent executor used by the Elastic AI Assistant. Main agent/chain that wraps the ActionsClientSimpleChatModel,
 * sets up a conversation BufferMemory from chat history, and registers tools like the ESQLKnowledgeBaseTool.
 *
 */
const callAgentExecutor = async ({
  abortSignal,
  actionsClient,
  alertsIndexPattern,
  assistantTools = [],
  bedrockChatEnabled,
  connectorId,
  esClient,
  esStore,
  langChainMessages,
  llmType,
  logger,
  isStream = false,
  onLlmResponse,
  onNewReplacements,
  replacements,
  request,
  size,
  traceOptions,
  dataClients,
  conversationId
}) => {
  var _dataClients$anonymiz, _traceOptions$project;
  const isOpenAI = llmType === 'openai';
  const llmClass = (0, _utils.getLlmClass)(llmType, bedrockChatEnabled);
  const llm = new llmClass({
    actionsClient,
    connectorId,
    llmType,
    logger,
    // possible client model override,
    // let this be undefined otherwise so the connector handles the model
    model: request.body.model,
    // ensure this is defined because we default to it in the language_models
    // This is where the LangSmith logs (Metadata > Invocation Params) are set
    temperature: (0, _server2.getDefaultArguments)(llmType).temperature,
    signal: abortSignal,
    streaming: isStream,
    // prevents the agent from retrying on failure
    // failure could be due to bad connector, we should deliver that result to the client asap
    maxRetries: 0
  });
  const anonymizationFieldsRes = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$anonymiz = dataClients.anonymizationFieldsDataClient) === null || _dataClients$anonymiz === void 0 ? void 0 : _dataClients$anonymiz.findDocuments({
    perPage: 1000,
    page: 1
  }));
  const anonymizationFields = anonymizationFieldsRes ? (0, _helpers.transformESSearchToAnonymizationFields)(anonymizationFieldsRes.data) : undefined;
  const pastMessages = langChainMessages.slice(0, -1); // all but the last message
  const latestMessage = langChainMessages.slice(-1); // the last message

  const memory = new _memory.BufferMemory({
    chatHistory: new _memory.ChatMessageHistory(pastMessages),
    memoryKey: 'chat_history',
    // this is the key expected by https://github.com/langchain-ai/langchainjs/blob/a13a8969345b0f149c1ca4a120d63508b06c52a5/langchain/src/agents/initialize.ts#L166
    inputKey: 'input',
    outputKey: 'output',
    returnMessages: true
  });
  const modelExists = await esStore.isModelInstalled();

  // Create a chain that uses the ELSER backed ElasticsearchStore, override k=10 for esql query generation for now
  const chain = _chains.RetrievalQAChain.fromLLM(llm, esStore.asRetriever(10));

  // Fetch any applicable tools that the source plugin may have registered
  const assistantToolParams = {
    alertsIndexPattern,
    anonymizationFields,
    chain,
    esClient,
    isEnabledKnowledgeBase: true,
    llm,
    logger,
    modelExists,
    onNewReplacements,
    replacements,
    request,
    size
  };
  const tools = assistantTools.flatMap(tool => {
    var _tool$getTool;
    return (_tool$getTool = tool.getTool(assistantToolParams)) !== null && _tool$getTool !== void 0 ? _tool$getTool : [];
  });
  logger.debug(`applicable tools: ${JSON.stringify(tools.map(t => t.name).join(', '), null, 2)}`);
  const executorArgs = {
    memory,
    verbose: false,
    handleParsingErrors: 'Try again, paying close attention to the allowed tool input'
  };
  // isOpenAI check is not on agentType alone because typescript doesn't like
  const executor = isOpenAI ? await (0, _agents.initializeAgentExecutorWithOptions)(tools, llm, {
    agentType: 'openai-functions',
    ...executorArgs
  }) : llmType === 'bedrock' && bedrockChatEnabled ? new _agents.AgentExecutor({
    agent: await (0, _agents.createToolCallingAgent)({
      llm,
      tools,
      prompt: _prompts.ChatPromptTemplate.fromMessages([['system', 'You are a helpful assistant'], ['placeholder', '{chat_history}'], ['human', '{input}'], ['placeholder', '{agent_scratchpad}']]),
      streamRunnable: isStream
    }),
    tools
  }) : await (0, _agents.initializeAgentExecutorWithOptions)(tools, llm, {
    agentType: 'structured-chat-zero-shot-react-description',
    ...executorArgs,
    returnIntermediateSteps: false,
    agentArgs: {
      // this is important to help LangChain correctly format tool input
      humanMessageTemplate: `Remember, when you have enough information, always prefix your final JSON output with "Final Answer:"\n\nQuestion: {input}\n\n{agent_scratchpad}.`,
      memoryPrompts: [new _prompts.MessagesPlaceholder('chat_history')],
      suffix: 'Begin! Reminder to ALWAYS use the above format, and to use tools if appropriate.'
    }
  });

  // Sets up tracer for tracing executions to APM. See x-pack/plugins/elastic_assistant/server/lib/langchain/tracers/README.mdx
  // If LangSmith env vars are set, executions will be traced there as well. See https://docs.smith.langchain.com/tracing
  const apmTracer = new _apm.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  let traceData;
  if (isStream) {
    var _traceOptions$tracers, _traceOptions$tags;
    let streamingSpan;
    if (_elasticApmNode.default.isStarted()) {
      var _agent$startSpan;
      streamingSpan = (_agent$startSpan = _elasticApmNode.default.startSpan(`${DEFAULT_AGENT_EXECUTOR_ID} (Streaming)`)) !== null && _agent$startSpan !== void 0 ? _agent$startSpan : undefined;
    }
    const {
      end: streamEnd,
      push,
      responseWithHeaders
    } = (0, _server.streamFactory)(request.headers, logger, false, false);
    let didEnd = false;
    const handleStreamEnd = (finalResponse, isError = false) => {
      var _streamingSpan3, _streamingSpan4, _streamingSpan5;
      if (onLlmResponse) {
        var _streamingSpan, _streamingSpan$transa, _streamingSpan$transa2, _streamingSpan2, _streamingSpan2$ids;
        onLlmResponse(finalResponse, {
          transactionId: (_streamingSpan = streamingSpan) === null || _streamingSpan === void 0 ? void 0 : (_streamingSpan$transa = _streamingSpan.transaction) === null || _streamingSpan$transa === void 0 ? void 0 : (_streamingSpan$transa2 = _streamingSpan$transa.ids) === null || _streamingSpan$transa2 === void 0 ? void 0 : _streamingSpan$transa2['transaction.id'],
          traceId: (_streamingSpan2 = streamingSpan) === null || _streamingSpan2 === void 0 ? void 0 : (_streamingSpan2$ids = _streamingSpan2.ids) === null || _streamingSpan2$ids === void 0 ? void 0 : _streamingSpan2$ids['trace.id']
        }, isError).catch(() => {});
      }
      streamEnd();
      didEnd = true;
      if (streamingSpan && !((_streamingSpan3 = streamingSpan) !== null && _streamingSpan3 !== void 0 && _streamingSpan3.outcome) || ((_streamingSpan4 = streamingSpan) === null || _streamingSpan4 === void 0 ? void 0 : _streamingSpan4.outcome) === 'unknown') {
        streamingSpan.outcome = 'success';
      }
      (_streamingSpan5 = streamingSpan) === null || _streamingSpan5 === void 0 ? void 0 : _streamingSpan5.end();
    };
    let message = '';
    let tokenParentRunId = '';
    executor.invoke({
      input: latestMessage[0].content,
      chat_history: [],
      signal: abortSignal
    }, {
      callbacks: [{
        handleLLMNewToken(payload, _idx, _runId, parentRunId) {
          if (tokenParentRunId.length === 0 && !!parentRunId) {
            // set the parent run id as the parentRunId of the first token
            // this is used to ensure that all tokens in the stream are from the same run
            // filtering out runs that are inside e.g. tool calls
            tokenParentRunId = parentRunId;
          }
          if (payload.length && !didEnd && tokenParentRunId === parentRunId) {
            push({
              payload,
              type: 'content'
            });
            // store message in case of error
            message += payload;
          }
        },
        handleChainEnd(outputs, runId, parentRunId) {
          // if parentRunId is undefined, this is the end of the stream
          if (!parentRunId) {
            handleStreamEnd(outputs.output);
          }
        }
      }, apmTracer, ...((_traceOptions$tracers = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers !== void 0 ? _traceOptions$tracers : [])],
      runName: DEFAULT_AGENT_EXECUTOR_ID,
      tags: (_traceOptions$tags = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tags) !== null && _traceOptions$tags !== void 0 ? _traceOptions$tags : []
    }).catch(err => {
      // if I throw an error here, it crashes the server. Not sure how to get around that.
      // If I put await on this function the error works properly, but when there is not an error
      // it waits for the entire stream to complete before resolving
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (error.message === 'AbortError') {
        // user aborted the stream, we must end it manually here
        return handleStreamEnd(message);
      }
      logger.error(`Error streaming from LangChain: ${error.message}`);
      push({
        payload: error.message,
        type: 'content'
      });
      handleStreamEnd(error.message, true);
    });
    return responseWithHeaders;
  }

  // Wrap executor call with an APM span for instrumentation
  const langChainResponse = await (0, _with_assistant_span.withAssistantSpan)(DEFAULT_AGENT_EXECUTOR_ID, async span => {
    var _span$transaction, _traceOptions$tracers2, _traceOptions$tags2;
    if ((span === null || span === void 0 ? void 0 : (_span$transaction = span.transaction) === null || _span$transaction === void 0 ? void 0 : _span$transaction.ids['transaction.id']) != null && (span === null || span === void 0 ? void 0 : span.ids['trace.id']) != null) {
      traceData = {
        // Transactions ID since this span is the parent
        transaction_id: span.transaction.ids['transaction.id'],
        trace_id: span.ids['trace.id']
      };
      span.addLabels({
        evaluationId: traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.evaluationId
      });
    }
    return executor.call({
      input: latestMessage[0].content
    }, {
      callbacks: [apmTracer, ...((_traceOptions$tracers2 = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers2 !== void 0 ? _traceOptions$tracers2 : [])],
      runName: DEFAULT_AGENT_EXECUTOR_ID,
      tags: (_traceOptions$tags2 = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tags) !== null && _traceOptions$tags2 !== void 0 ? _traceOptions$tags2 : []
    });
  });
  const langChainOutput = langChainResponse.output;
  if (onLlmResponse) {
    await onLlmResponse(langChainOutput, traceData);
  }
  return {
    body: {
      connector_id: connectorId,
      data: langChainOutput,
      // the response from the actions framework
      trace_data: traceData,
      replacements,
      status: 'ok',
      conversationId
    },
    headers: {
      'content-type': 'application/json'
    }
  };
};
exports.callAgentExecutor = callAgentExecutor;