"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TERMS_QUERY_SIZE = exports.FALLBACK_SIMILARITY_SEARCH_SIZE = exports.ElasticsearchStore = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _vectorstores = require("@langchain/core/vectorstores");
var uuid = _interopRequireWildcard(require("uuid"));
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticsearch_embeddings = require("../embeddings/elasticsearch_embeddings");
var _get_flattened_hits = require("./helpers/get_flattened_hits");
var _get_msearch_query_body = require("./helpers/get_msearch_query_body");
var _get_terms_search_query = require("./helpers/get_terms_search_query");
var _get_vector_search_query = require("./helpers/get_vector_search_query");
var _constants = require("../../../routes/knowledge_base/constants");
var _get_required_kb_docs_terms_query_dsl = require("./helpers/get_required_kb_docs_terms_query_dsl");
var _event_based_telemetry = require("../../telemetry/event_based_telemetry");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A fallback for the query `size` that determines how many documents to
 * return from Elasticsearch when performing a similarity search.
 *
 * The size is typically determined by the implementation of LangChain's
 * `VectorStoreRetriever._getRelevantDocuments` function, so this fallback is
 * only required when using the `ElasticsearchStore` directly.
 */
const FALLBACK_SIMILARITY_SEARCH_SIZE = exports.FALLBACK_SIMILARITY_SEARCH_SIZE = 10;

/** The maximum number of hits to return from a `terms` query, via the `size` parameter */
const TERMS_QUERY_SIZE = exports.TERMS_QUERY_SIZE = 10000;

/**
 * Basic ElasticsearchStore implementation only leveraging ELSER for storage and retrieval.
 */
class ElasticsearchStore extends _vectorstores.VectorStore {
  _vectorstoreType() {
    return 'elasticsearch';
  }
  constructor(esClient, _index, logger, telemetry, model, kbResource, kbDataClient) {
    super(new _elasticsearch_embeddings.ElasticsearchEmbeddings(logger), {
      esClient,
      index: _index
    });
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "kbDataClient", void 0);
    (0, _defineProperty2.default)(this, "index", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "telemetry", void 0);
    (0, _defineProperty2.default)(this, "model", void 0);
    (0, _defineProperty2.default)(this, "kbResource", void 0);
    /**
     * Add documents to the store. Embeddings are created on ingest into index configured with
     * ELSER ingest pipeline. Returns a list of document IDs.
     *
     * @param documents Documents to add to the store
     * @param options Any additional options as defined in the interface
     * @returns Promise<string[]> of document IDs added to the store
     */
    (0, _defineProperty2.default)(this, "addDocuments", async (documents, options) => {
      // Code path for when `assistantKnowledgeBaseByDefault` FF is enabled
      // Once removed replace addDocuments() w/ addDocumentsViaDataClient()
      if (this.kbDataClient != null) {
        return this.addDocumentsViaDataClient(documents, options);
      }
      const pipelineExists = await this.pipelineExists();
      if (!pipelineExists) {
        await this.createPipeline();
      }
      const operations = documents.flatMap(({
        pageContent,
        metadata
      }) => [{
        index: {
          _index: this.index,
          _id: uuid.v4()
        }
      }, {
        text: pageContent,
        metadata
      }]);
      try {
        const response = await this.esClient.bulk({
          refresh: true,
          operations
        });
        this.logger.debug(`Add Documents Response:\n ${JSON.stringify(response)}`);
        const errorIds = response.items.filter(i => {
          var _i$index;
          return ((_i$index = i.index) === null || _i$index === void 0 ? void 0 : _i$index.error) != null;
        });
        operations.forEach((op, i) => {
          if (errorIds.some(e => {
            var _e$index, _op$index;
            return ((_e$index = e.index) === null || _e$index === void 0 ? void 0 : _e$index._id) === ((_op$index = op.index) === null || _op$index === void 0 ? void 0 : _op$index._id);
          })) {
            this.logger.error(`Error adding document to KB: ${JSON.stringify(operations === null || operations === void 0 ? void 0 : operations[i + 1])}`);
          }
        });
        return response.items.flatMap(i => {
          var _i$index2;
          return ((_i$index2 = i.index) === null || _i$index2 === void 0 ? void 0 : _i$index2._id) != null && i.index.error == null ? [i.index._id] : [];
        });
      } catch (e) {
        this.logger.error(`Error loading data into KB\n ${e}`);
        return [];
      }
    });
    (0, _defineProperty2.default)(this, "addDocumentsViaDataClient", async (documents, options) => {
      if (!this.kbDataClient) {
        this.logger.error('No kbDataClient provided');
        return [];
      }
      try {
        const response = await this.kbDataClient.addKnowledgeBaseDocuments({
          documents
        });
        return response.map(doc => doc.id);
      } catch (e) {
        this.logger.error(`Error loading data into KB\n ${e}`);
        return [];
      }
    });
    /**
     * Add vectors to the store. Returns a list of document IDs.
     *
     * @param vectors Vector representation of documents to add to the store
     * @param documents Documents corresponding to the provided vectors
     * @param options Any additional options as defined in the interface
     * @returns Promise<string[]> of document IDs added to the store
     */
    (0, _defineProperty2.default)(this, "addVectors", (vectors, documents, options) => {
      // Note: implement if/when needed
      this.logger.info('ElasticsearchStore.addVectors not implemented');
      return Promise.resolve(undefined);
    });
    /**
     * Performs similarity search on the store using the provided query vector and filter, returning k similar
     * documents along with their score.
     *
     * @param query Query vector to search with
     * @param k Number of similar documents to return
     * @param filter Optional filter to apply to the search
     *
     *  @returns Promise<Array<[Document, number]>> of similar documents and their scores
     */
    (0, _defineProperty2.default)(this, "similaritySearchVectorWithScore", (query, k, filter) => {
      // Note: Implement if needed
      this.logger.info('ElasticsearchStore.similaritySearchVectorWithScore not implemented');
      return Promise.resolve([]);
    });
    // Non-abstract function overrides
    /**
     * Performs similarity search on the store using the provided query string and filter, returning k similar
     * @param query Query vector to search with
     * @param k Number of similar documents to return
     * @param filter Optional filter to apply to the search
     * @param _callbacks Optional callbacks
     *
     * Fun facts:
     * - This function is called by LangChain's `VectorStoreRetriever._getRelevantDocuments`
     * - The `k` parameter is typically determined by LangChain's `VectorStoreRetriever._getRelevantDocuments`, and has been observed to default to `4` in the wild (see langchain/dist/vectorstores/base.ts)
     * @returns Promise<Document[]> of similar documents
     */
    (0, _defineProperty2.default)(this, "similaritySearch", async (query, k, filter, _callbacks) => {
      // requiredDocs is an array of filters that can be used in a `bool` Elasticsearch DSL query to filter in/out required KB documents:
      const requiredDocs = (0, _get_required_kb_docs_terms_query_dsl.getRequiredKbDocsTermsQueryDsl)(this.kbResource);

      // The `k` parameter is typically provided by LangChain's `VectorStoreRetriever._getRelevantDocuments`, which calls this function:
      const vectorSearchQuerySize = k !== null && k !== void 0 ? k : FALLBACK_SIMILARITY_SEARCH_SIZE;

      // build a vector search query:
      const vectorSearchQuery = (0, _get_vector_search_query.getVectorSearchQuery)({
        filter,
        modelId: this.model,
        mustNotTerms: requiredDocs,
        query
      });

      // build a (separate) terms search query:
      const termsSearchQuery = (0, _get_terms_search_query.getTermsSearchQuery)(requiredDocs);

      // combine the vector search query and the terms search queries into a single multi-search query:
      const mSearchQueryBody = (0, _get_msearch_query_body.getMsearchQueryBody)({
        index: this.index,
        termsSearchQuery,
        termsSearchQuerySize: TERMS_QUERY_SIZE,
        vectorSearchQuery,
        vectorSearchQuerySize
      });
      try {
        var _result$took;
        // execute both queries via a single multi-search request:
        const result = await this.esClient.msearch(mSearchQueryBody);

        // flatten the results of the combined queries into a single array of hits:
        const results = result.responses.flatMap(response => {
          const maybeEsqlMsearchResponse = response;
          return (0, _get_flattened_hits.getFlattenedHits)(maybeEsqlMsearchResponse);
        });
        this.telemetry.reportEvent(_event_based_telemetry.KNOWLEDGE_BASE_EXECUTION_SUCCESS_EVENT.eventType, {
          model: this.model,
          resourceAccessed: this.kbResource,
          resultCount: results.length,
          responseTime: (_result$took = result.took) !== null && _result$took !== void 0 ? _result$took : 0
        });
        this.logger.debug(`Similarity search metadata source:\n${JSON.stringify(results.map(r => {
          var _r$metadata$source, _r$metadata;
          return (_r$metadata$source = r === null || r === void 0 ? void 0 : (_r$metadata = r.metadata) === null || _r$metadata === void 0 ? void 0 : _r$metadata.source) !== null && _r$metadata$source !== void 0 ? _r$metadata$source : '(missing metadata.source)';
        }), null, 2)}`);
        return results;
      } catch (e) {
        const error = (0, _securitysolutionEsUtils.transformError)(e);
        this.telemetry.reportEvent(_event_based_telemetry.KNOWLEDGE_BASE_EXECUTION_ERROR_EVENT.eventType, {
          model: this.model,
          resourceAccessed: this.kbResource,
          errorMessage: error.message
        });
        this.logger.error(e);
        return [];
      }
    });
    // ElasticsearchStore explicit utility functions
    /**
     * Checks if the provided index exists in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the index exists
     * @param index Index to check
     * @returns Promise<boolean> indicating whether the index exists
     */
    (0, _defineProperty2.default)(this, "indexExists", async index => {
      return this.esClient.indices.exists({
        index: index !== null && index !== void 0 ? index : this.index
      });
    });
    /**
     * Create index for ELSER embeddings in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the index was created
     */
    (0, _defineProperty2.default)(this, "createIndex", async ({
      index,
      pipeline
    } = {}) => {
      const mappings = {
        properties: {
          metadata: {
            properties: {
              /** the category of knowledge, e.g. `esql` */
              kbResource: {
                type: 'keyword'
              },
              /** when `true`, return this document in all searches for the `kbResource` */
              required: {
                type: 'boolean'
              },
              /** often a file path when the document was created via a LangChain `DirectoryLoader`, this metadata describes the origin of the document */
              source: {
                type: 'keyword'
              }
            }
          },
          vector: {
            properties: {
              tokens: {
                type: 'rank_features'
              }
            }
          }
        }
      };
      const settings = {
        default_pipeline: pipeline !== null && pipeline !== void 0 ? pipeline : _constants.KNOWLEDGE_BASE_INGEST_PIPELINE
      };
      const response = await this.esClient.indices.create({
        index: index !== null && index !== void 0 ? index : this.index,
        mappings,
        settings
      });
      return response.acknowledged;
    });
    /**
     * Delete index for ELSER embeddings in Elasticsearch
     * @param index Index to delete, otherwise uses the default index
     *
     * @returns Promise<boolean> indicating whether the index was created
     */
    (0, _defineProperty2.default)(this, "deleteIndex", async index => {
      // Code path for when `assistantKnowledgeBaseByDefault` FF is enabled
      // We won't be supporting delete operations for the KB data stream going forward, so this can be removed along with the FF
      if (this.kbDataClient != null) {
        const response = await this.esClient.indices.deleteDataStream({
          name: index !== null && index !== void 0 ? index : this.index
        });
        return response.acknowledged;
      }
      const response = await this.esClient.indices.delete({
        index: index !== null && index !== void 0 ? index : this.index
      });
      return response.acknowledged;
    });
    /**
     * Checks if the provided ingest pipeline exists in Elasticsearch
     *
     * @param pipelineId ID of the ingest pipeline to check
     * @returns Promise<boolean> indicating whether the pipeline exists
     */
    (0, _defineProperty2.default)(this, "pipelineExists", async pipelineId => {
      try {
        var _ref, _this$kbDataClient;
        const id = (_ref = pipelineId !== null && pipelineId !== void 0 ? pipelineId : (_this$kbDataClient = this.kbDataClient) === null || _this$kbDataClient === void 0 ? void 0 : _this$kbDataClient.options.ingestPipelineResourceName) !== null && _ref !== void 0 ? _ref : _constants.KNOWLEDGE_BASE_INGEST_PIPELINE;
        const response = await this.esClient.ingest.getPipeline({
          id
        });
        return Object.keys(response).length > 0;
      } catch (e) {
        // The GET /_ingest/pipeline/{pipelineId} API returns an empty object w/ 404 Not Found.
        return false;
      }
    });
    /**
     * Create ingest pipeline for ELSER in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the pipeline was created
     */
    (0, _defineProperty2.default)(this, "createPipeline", async ({
      id,
      description
    } = {}) => {
      var _ref2, _this$kbDataClient2;
      const response = await this.esClient.ingest.putPipeline({
        id: (_ref2 = id !== null && id !== void 0 ? id : (_this$kbDataClient2 = this.kbDataClient) === null || _this$kbDataClient2 === void 0 ? void 0 : _this$kbDataClient2.options.ingestPipelineResourceName) !== null && _ref2 !== void 0 ? _ref2 : _constants.KNOWLEDGE_BASE_INGEST_PIPELINE,
        description: description !== null && description !== void 0 ? description : 'Embedding pipeline for Elastic AI Assistant ELSER Knowledge Base',
        processors: [{
          inference: {
            model_id: this.model,
            target_field: 'vector',
            field_map: {
              text: 'text_field'
            },
            inference_config: {
              // @ts-expect-error
              text_expansion: {
                results_field: 'tokens'
              }
            }
          }
        }]
      });
      return response.acknowledged;
    });
    /**
     * Delete ingest pipeline for ELSER in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the pipeline was created
     */
    (0, _defineProperty2.default)(this, "deletePipeline", async pipelineId => {
      var _ref3, _this$kbDataClient3;
      const response = await this.esClient.ingest.deletePipeline({
        id: (_ref3 = pipelineId !== null && pipelineId !== void 0 ? pipelineId : (_this$kbDataClient3 = this.kbDataClient) === null || _this$kbDataClient3 === void 0 ? void 0 : _this$kbDataClient3.options.ingestPipelineResourceName) !== null && _ref3 !== void 0 ? _ref3 : _constants.KNOWLEDGE_BASE_INGEST_PIPELINE
      });
      return response.acknowledged;
    });
    this.esClient = esClient;
    this.index = _index !== null && _index !== void 0 ? _index : _constants.KNOWLEDGE_BASE_INDEX_PATTERN;
    this.logger = logger;
    this.telemetry = telemetry;
    this.model = model !== null && model !== void 0 ? model : '.elser_model_2';
    this.kbResource = kbResource !== null && kbResource !== void 0 ? kbResource : _constants.ESQL_RESOURCE;
    this.kbDataClient = kbDataClient;
  }
  /**
   * Checks if the provided model is installed in Elasticsearch
   *
   * @param modelId ID of the model to check
   * @returns Promise<boolean> indicating whether the model is installed
   */
  async isModelInstalled(modelId) {
    try {
      // Code path for when `assistantKnowledgeBaseByDefault` FF is enabled
      if (this.kbDataClient != null) {
        // esStore.isModelInstalled() is actually checking if the model is deployed, not installed, so do that instead
        return this.kbDataClient.isModelDeployed();
      }
      const getResponse = await this.esClient.ml.getTrainedModelsStats({
        model_id: modelId !== null && modelId !== void 0 ? modelId : this.model
      });
      this.logger.debug(`modelId: ${modelId}`);

      // For standardized way of checking deployment status see: https://github.com/elastic/elasticsearch/issues/106986
      const isReadyESS = stats => {
        var _stats$deployment_sta, _stats$deployment_sta2;
        return ((_stats$deployment_sta = stats.deployment_stats) === null || _stats$deployment_sta === void 0 ? void 0 : _stats$deployment_sta.state) === 'started' && ((_stats$deployment_sta2 = stats.deployment_stats) === null || _stats$deployment_sta2 === void 0 ? void 0 : _stats$deployment_sta2.allocation_status.state) === 'fully_allocated';
      };
      const isReadyServerless = stats => {
        var _stats$deployment_sta3;
        return ((_stats$deployment_sta3 = stats.deployment_stats) === null || _stats$deployment_sta3 === void 0 ? void 0 : _stats$deployment_sta3.nodes).some(node => node.routing_state.routing_state === 'started');
      };
      return getResponse.trained_model_stats.some(stats => isReadyESS(stats) || isReadyServerless(stats));
    } catch (e) {
      // Returns 404 if it doesn't exist
      return false;
    }
  }
}
exports.ElasticsearchStore = ElasticsearchStore;