"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.pipelineExists = exports.deletePipeline = exports.createPipeline = exports.createGetElserId = void 0;
var _ingest_pipeline = require("../ai_assistant_data_clients/knowledge_base/ingest_pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates a function that returns the ELSER model ID
 *
 * @param ml
 */
const createGetElserId = trainedModelsProvider => async () =>
// Force check to happen as internal user
(await trainedModelsProvider({}, {}).getELSER()).model_id;
exports.createGetElserId = createGetElserId;
/**
 * Checks if the provided ingest pipeline exists in Elasticsearch
 *
 * @param params params
 * @param params.esClient Elasticsearch client with privileges to check for ingest pipelines
 * @param params.id ID of the ingest pipeline to check
 *
 * @returns Promise<boolean> indicating whether the pipeline exists
 */
const pipelineExists = async ({
  esClient,
  id
}) => {
  try {
    const response = await esClient.ingest.getPipeline({
      id
    });
    return Object.keys(response).length > 0;
  } catch (e) {
    // The GET /_ingest/pipeline/{pipelineId} API returns an empty object w/ 404 Not Found.
    return false;
  }
};
exports.pipelineExists = pipelineExists;
/**
 * Create ingest pipeline for ELSER in Elasticsearch
 *
 * @param params params
 * @param params.esClient Elasticsearch client with privileges to check for ingest pipelines
 * @param params.id ID of the ingest pipeline
 * @param params.modelId ID of the ELSER model
 *
 * @returns Promise<boolean> indicating whether the pipeline was created
 */
const createPipeline = async ({
  esClient,
  id,
  modelId
}) => {
  try {
    const response = await esClient.ingest.putPipeline((0, _ingest_pipeline.knowledgeBaseIngestPipeline)({
      id,
      modelId
    }));
    return response.acknowledged;
  } catch (e) {
    return false;
  }
};
exports.createPipeline = createPipeline;
/**
 * Delete ingest pipeline for ELSER in Elasticsearch
 *
 * @returns Promise<boolean> indicating whether the pipeline was created
 */
const deletePipeline = async ({
  esClient,
  id
}) => {
  const response = await esClient.ingest.deletePipeline({
    id
  });
  return response.acknowledged;
};
exports.deletePipeline = deletePipeline;