"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AIAssistantKnowledgeBaseDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _document = require("langchain/document");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _ = require("..");
var _esql_loader = require("../../lib/langchain/content_loaders/esql_loader");
var _create_knowledge_base_entry = require("./create_knowledge_base_entry");
var _transforms = require("./transforms");
var _constants = require("../../routes/knowledge_base/constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AIAssistantKnowledgeBaseDataClient extends _.AIAssistantDataClient {
  constructor(options) {
    super(options);
    /**
     * Returns whether setup of the Knowledge Base can be performed (essentially an ML features check)
     *
     */
    (0, _defineProperty2.default)(this, "isSetupAvailable", async () => {
      // ML plugin requires request to retrieve capabilities, which are in turn scoped to the user from the request,
      // so we just test the API for a 404 instead to determine if ML is 'available'
      // TODO: expand to include memory check, see https://github.com/elastic/ml-team/issues/1208#issuecomment-2115770318
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        await esClient.ml.getMemoryStats({
          human: true
        });
      } catch (error) {
        return false;
      }
      return true;
    });
    /**
     * Downloads and installs ELSER model if not already installed
     *
     * @param soClient SavedObjectsClientContract for installing ELSER so that ML SO's are in sync
     */
    (0, _defineProperty2.default)(this, "installModel", async ({
      soClient
    }) => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Installing ELSER model '${elserId}'...`);
      try {
        await this.options.ml
        // TODO: Potentially plumb soClient through DataClient from pluginStart
        .trainedModelsProvider({}, soClient).installElasticModel(elserId);
      } catch (error) {
        this.options.logger.error(`Error installing ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Returns whether ELSER is installed/ready to deploy
     *
     * @returns Promise<boolean> indicating whether the model is installed
     */
    (0, _defineProperty2.default)(this, "isModelInstalled", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Checking if ELSER model '${elserId}' is installed...`);
      try {
        var _getResponse$trained_;
        const esClient = await this.options.elasticsearchClientPromise;
        const getResponse = await esClient.ml.getTrainedModels({
          model_id: elserId,
          include: 'definition_status'
        });
        return Boolean((_getResponse$trained_ = getResponse.trained_model_configs[0]) === null || _getResponse$trained_ === void 0 ? void 0 : _getResponse$trained_.fully_defined);
      } catch (error) {
        if (!(0, _helpers.isModelAlreadyExistsError)(error)) {
          this.options.logger.error(`Error checking if ELSER model '${elserId}' is installed:\n${error}`);
        }
        return false;
      }
    });
    /**
     * Deploy the ELSER model with default configuration
     */
    (0, _defineProperty2.default)(this, "deployModel", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Deploying ELSER model '${elserId}'...`);
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        await esClient.ml.startTrainedModelDeployment({
          model_id: elserId,
          wait_for: 'fully_allocated'
        });
      } catch (error) {
        this.options.logger.error(`Error deploying ELSER model '${elserId}':\n${error}`);
        throw new Error(`Error deploying ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Checks if the provided model is deployed and allocated in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the model is deployed
     */
    (0, _defineProperty2.default)(this, "isModelDeployed", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Checking if ELSER model '${elserId}' is deployed...`);
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        const getResponse = await esClient.ml.getTrainedModelsStats({
          model_id: elserId
        });

        // For standardized way of checking deployment status see: https://github.com/elastic/elasticsearch/issues/106986
        const isReadyESS = stats => {
          var _stats$deployment_sta, _stats$deployment_sta2;
          return ((_stats$deployment_sta = stats.deployment_stats) === null || _stats$deployment_sta === void 0 ? void 0 : _stats$deployment_sta.state) === 'started' && ((_stats$deployment_sta2 = stats.deployment_stats) === null || _stats$deployment_sta2 === void 0 ? void 0 : _stats$deployment_sta2.allocation_status.state) === 'fully_allocated';
        };
        const isReadyServerless = stats => {
          var _stats$deployment_sta3;
          return ((_stats$deployment_sta3 = stats.deployment_stats) === null || _stats$deployment_sta3 === void 0 ? void 0 : _stats$deployment_sta3.nodes).some(node => node.routing_state.routing_state === 'started');
        };
        return getResponse.trained_model_stats.some(stats => isReadyESS(stats) || isReadyServerless(stats));
      } catch (e) {
        // Returns 404 if it doesn't exist
        return false;
      }
    });
    /**
     * Downloads and deploys recommended ELSER (if not already), then loads ES|QL docs
     *
     * NOTE: Before automatically installing ELSER in the background, we should perform deployment resource checks
     * Only necessary for ESS, as Serverless can always auto-install if `productTier === complete`
     * See ml-team issue for providing 'dry run' flag to perform these checks: https://github.com/elastic/ml-team/issues/1208
     *
     * @param options
     * @param options.esStore ElasticsearchStore for loading ES|QL docs via LangChain loaders
     * @param options.soClient SavedObjectsClientContract for installing ELSER so that ML SO's are in sync
     *
     * @returns Promise<void>
     */
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", async ({
      esStore,
      soClient
    }) => {
      if (this.options.getIsKBSetupInProgress()) {
        this.options.logger.debug('Knowledge Base setup already in progress');
        return;
      }
      this.options.logger.debug('Starting Knowledge Base setup...');
      this.options.setIsKBSetupInProgress(true);
      const elserId = await this.options.getElserId();
      try {
        const isInstalled = await this.isModelInstalled();
        if (!isInstalled) {
          await this.installModel({
            soClient
          });
          await (0, _pRetry.default)(async () => (await this.isModelInstalled()) ? Promise.resolve() : Promise.reject(new Error('Model not installed')), {
            minTimeout: 10000,
            maxTimeout: 10000,
            retries: 10
          });
          this.options.logger.debug(`ELSER model '${elserId}' successfully installed!`);
        } else {
          this.options.logger.debug(`ELSER model '${elserId}' is already installed`);
        }
        const isDeployed = await this.isModelDeployed();
        if (!isDeployed) {
          await this.deployModel();
          await (0, _pRetry.default)(async () => (await this.isModelDeployed()) ? Promise.resolve() : Promise.reject(new Error('Model not deployed')), {
            minTimeout: 2000,
            retries: 10
          });
          this.options.logger.debug(`ELSER model '${elserId}' successfully deployed!`);
        } else {
          this.options.logger.debug(`ELSER model '${elserId}' is already deployed`);
        }
        this.options.logger.debug(`Checking if Knowledge Base docs have been loaded...`);
        const kbDocsLoaded = (await esStore.similaritySearch(_constants.ESQL_DOCS_LOADED_QUERY)).length > 0;
        if (!kbDocsLoaded) {
          this.options.logger.debug(`Loading KB docs...`);
          await (0, _esql_loader.loadESQL)(esStore, this.options.logger);
        } else {
          this.options.logger.debug(`Knowledge Base docs already loaded!`);
        }
      } catch (e) {
        this.options.setIsKBSetupInProgress(false);
        this.options.logger.error(`Error setting up Knowledge Base: ${e.message}`);
        throw new Error(`Error setting up Knowledge Base: ${e.message}`);
      }
      this.options.setIsKBSetupInProgress(false);
    });
    /**
     * Adds LangChain Documents to the knowledge base
     *
     * @param {Array<Document<Metadata>>} documents - LangChain Documents to add to the knowledge base
     */
    (0, _defineProperty2.default)(this, "addKnowledgeBaseDocuments", async ({
      documents
    }) => {
      var _created$data$hits$hi;
      const writer = await this.getWriter();
      const changedAt = new Date().toISOString();
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      // @ts-ignore
      const {
        errors,
        docs_created: docsCreated
      } = await writer.bulk({
        documentsToCreate: documents.map(doc => {
          var _doc$metadata$kbResou, _doc$metadata$require, _doc$metadata$source;
          return (0, _create_knowledge_base_entry.transformToCreateSchema)(changedAt, this.spaceId, authenticatedUser, {
            metadata: {
              kbResource: (_doc$metadata$kbResou = doc.metadata.kbResource) !== null && _doc$metadata$kbResou !== void 0 ? _doc$metadata$kbResou : 'unknown',
              required: (_doc$metadata$require = doc.metadata.required) !== null && _doc$metadata$require !== void 0 ? _doc$metadata$require : false,
              source: (_doc$metadata$source = doc.metadata.source) !== null && _doc$metadata$source !== void 0 ? _doc$metadata$source : 'unknown'
            },
            text: doc.pageContent
          });
        }),
        authenticatedUser
      });
      const created = docsCreated.length > 0 ? await this.findDocuments({
        page: 1,
        perPage: 10000,
        filter: docsCreated.map(c => `_id:${c}`).join(' OR ')
      }) : undefined;
      this.options.logger.debug(`created: ${(_created$data$hits$hi = created === null || created === void 0 ? void 0 : created.data.hits.hits.length) !== null && _created$data$hits$hi !== void 0 ? _created$data$hits$hi : '0'}`);
      this.options.logger.debug(`errors: ${JSON.stringify(errors, null, 2)}`);
      return created !== null && created !== void 0 && created.data ? (0, _transforms.transformESSearchToKnowledgeBaseEntry)(created === null || created === void 0 ? void 0 : created.data) : [];
    });
    /**
     * Performs similarity search to retrieve LangChain Documents from the knowledge base
     */
    (0, _defineProperty2.default)(this, "getKnowledgeBaseDocuments", async ({
      filter,
      kbResource,
      query,
      required
    }) => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      const modelId = await this.options.getElserId();
      const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
        filter,
        kbResource,
        modelId,
        query,
        required,
        user
      });
      try {
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 10,
          query: vectorSearchQuery
        });
        const results = result.hits.hits.map(hit => {
          var _hit$_source$text, _hit$_source, _hit$_source$metadata, _hit$_source2;
          return new _document.Document({
            pageContent: (_hit$_source$text = hit === null || hit === void 0 ? void 0 : (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.text) !== null && _hit$_source$text !== void 0 ? _hit$_source$text : '',
            metadata: (_hit$_source$metadata = hit === null || hit === void 0 ? void 0 : (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.metadata) !== null && _hit$_source$metadata !== void 0 ? _hit$_source$metadata : {}
          });
        });
        this.options.logger.debug(`getKnowledgeBaseDocuments() - Similarity Search Query:\n ${JSON.stringify(vectorSearchQuery)}`);
        this.options.logger.debug(`getKnowledgeBaseDocuments() - Similarity Search Results:\n ${JSON.stringify(results)}`);
        return results;
      } catch (e) {
        this.options.logger.error(`Error performing KB Similarity Search: ${e.message}`);
        return [];
      }
    });
    /**
     * Creates a new Knowledge Base Entry.
     *
     * @param knowledgeBaseEntry
     */
    (0, _defineProperty2.default)(this, "createKnowledgeBaseEntry", async ({
      knowledgeBaseEntry
    }) => {
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      this.options.logger.debug(`Creating Knowledge Base Entry:\n ${JSON.stringify(knowledgeBaseEntry, null, 2)}`);
      this.options.logger.debug(`kbIndex: ${this.indexTemplateAndPattern.alias}`);
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _create_knowledge_base_entry.createKnowledgeBaseEntry)({
        esClient,
        knowledgeBaseIndex: this.indexTemplateAndPattern.alias,
        logger: this.options.logger,
        spaceId: this.spaceId,
        user: authenticatedUser,
        knowledgeBaseEntry
      });
    });
    this.options = options;
  }
  get isSetupInProgress() {
    return this.options.getIsKBSetupInProgress();
  }
}
exports.AIAssistantKnowledgeBaseDataClient = AIAssistantKnowledgeBaseDataClient;