"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToCreateScheme = exports.createAttackDiscovery = void 0;
var _uuid = require("uuid");
var _get_attack_discovery = require("./get_attack_discovery");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAttackDiscovery = async ({
  esClient,
  attackDiscoveryIndex,
  spaceId,
  user,
  attackDiscoveryCreate,
  logger
}) => {
  const createdAt = new Date().toISOString();
  const body = transformToCreateScheme(createdAt, spaceId, user, attackDiscoveryCreate);
  const id = (attackDiscoveryCreate === null || attackDiscoveryCreate === void 0 ? void 0 : attackDiscoveryCreate.id) || (0, _uuid.v4)();
  try {
    const response = await esClient.create({
      body,
      id,
      index: attackDiscoveryIndex,
      refresh: 'wait_for'
    });
    const createdAttackDiscovery = await (0, _get_attack_discovery.getAttackDiscovery)({
      esClient,
      attackDiscoveryIndex,
      id: response._id,
      logger,
      user
    });
    return createdAttackDiscovery;
  } catch (err) {
    logger.error(`Error creating attack discovery: ${err} with id: ${id}`);
    throw err;
  }
};
exports.createAttackDiscovery = createAttackDiscovery;
const transformToCreateScheme = (createdAt, spaceId, user, {
  attackDiscoveries,
  apiConfig,
  alertsContextCount,
  replacements,
  status
}) => {
  return {
    '@timestamp': createdAt,
    created_at: createdAt,
    users: [{
      id: user.profile_uid,
      name: user.username
    }],
    status,
    api_config: {
      action_type_id: apiConfig.actionTypeId,
      connector_id: apiConfig.connectorId,
      default_system_prompt_id: apiConfig.defaultSystemPromptId,
      model: apiConfig.model,
      provider: apiConfig.provider
    },
    alerts_context_count: alertsContextCount,
    attack_discoveries: attackDiscoveries === null || attackDiscoveries === void 0 ? void 0 : attackDiscoveries.map(attackDiscovery => {
      var _attackDiscovery$time;
      return {
        id: attackDiscovery.id,
        alert_ids: attackDiscovery.alertIds,
        title: attackDiscovery.title,
        details_markdown: attackDiscovery.detailsMarkdown,
        entity_summary_markdown: attackDiscovery.entitySummaryMarkdown,
        mitre_attack_tactics: attackDiscovery.mitreAttackTactics,
        summary_markdown: attackDiscovery.summaryMarkdown,
        timestamp: (_attackDiscovery$time = attackDiscovery.timestamp) !== null && _attackDiscovery$time !== void 0 ? _attackDiscovery$time : createdAt
      };
    }),
    updated_at: createdAt,
    last_viewed_at: createdAt,
    replacements: replacements ? Object.keys(replacements).map(key => ({
      uuid: key,
      value: replacements[key]
    })) : undefined,
    namespace: spaceId
  };
};
exports.transformToCreateScheme = transformToCreateScheme;