"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getQuery = exports.getIndexResultsLatestRoute = void 0;
var _constants = require("../../../common/constants");
var _build_response = require("../../lib/build_response");
var _common = require("../../schemas/common");
var _result = require("../../schemas/result");
var _translations = require("../../translations");
var _translations2 = require("./translations");
var _get_authorized_index_names = require("../../helpers/get_authorized_index_names");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getQuery = indexName => ({
  size: 0,
  query: {
    bool: {
      filter: [{
        terms: {
          indexName
        }
      }]
    }
  },
  aggs: {
    latest: {
      terms: {
        field: 'indexName',
        size: 10000
      },
      // big enough to get all indexNames, but under `index.max_terms_count` (default 65536)
      aggs: {
        latest_doc: {
          top_hits: {
            size: 1,
            sort: [{
              '@timestamp': {
                order: 'desc'
              }
            }]
          }
        }
      }
    }
  }
});
exports.getQuery = getQuery;
const getIndexResultsLatestRoute = (router, logger) => {
  router.versioned.get({
    path: _constants.GET_INDEX_RESULTS_LATEST,
    access: 'internal',
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidation)(_result.GetIndexResultsLatestParams)
      }
    }
  }, async (context, request, response) => {
    const services = await context.resolve(['core', 'dataQualityDashboard']);
    const resp = (0, _build_response.buildResponse)(response);
    let index;
    try {
      index = await services.dataQualityDashboard.getResultsIndexName();
    } catch (err) {
      logger.error(`[GET results] Error retrieving results index name: ${err.message}`);
      return resp.error({
        body: `${_translations2.API_RESULTS_INDEX_NOT_AVAILABLE}: ${err.message}`,
        statusCode: 503
      });
    }
    try {
      var _aggregations$latest$, _aggregations$latest;
      const {
        client
      } = services.core.elasticsearch;
      const {
        pattern
      } = request.params;
      const authorizedIndexNames = await (0, _get_authorized_index_names.getAuthorizedIndexNames)(client, pattern);
      if (authorizedIndexNames.length === 0) {
        return response.ok({
          body: []
        });
      }

      // Get the latest result for each indexName
      const query = {
        index,
        ...getQuery(authorizedIndexNames)
      };
      const {
        aggregations
      } = await client.asInternalUser.search(query);
      const results = (_aggregations$latest$ = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$latest = aggregations.latest) === null || _aggregations$latest === void 0 ? void 0 : _aggregations$latest.buckets.map(bucket => bucket.latest_doc.hits.hits[0]._source)) !== null && _aggregations$latest$ !== void 0 ? _aggregations$latest$ : [];
      return response.ok({
        body: results
      });
    } catch (err) {
      var _err$message, _err$statusCode;
      logger.error(err.message);
      return resp.error({
        body: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : _translations.API_DEFAULT_ERROR_MESSAGE,
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500
      });
    }
  });
};
exports.getIndexResultsLatestRoute = getIndexResultsLatestRoute;