"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FullStoryShipper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _format_payload = require("./format_payload");
var _load_snippet = require("./load_snippet");
var _get_parsed_version = require("./get_parsed_version");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const PAGE_VARS_KEYS = [
// Page-specific keys
'pageName', 'page', 'entityId', 'applicationId',
// Deployment-specific keys
'version',
// x4, split to version_major, version_minor, version_patch for easier filtering
'buildSha',
// Useful for Serverless
'cloudId', 'deploymentId', 'projectId',
// projectId and deploymentId are mutually exclusive. They shouldn't be sent in the same offering.
'cluster_name', 'cluster_uuid', 'cluster_version', 'labels.serverless', 'license_id', 'license_status', 'license_type',
// Session-specific
'session_id', 'preferred_languages'];

/**
 * FullStory shipper configuration.
 */

/**
 * FullStory shipper.
 */
class FullStoryShipper {
  /**
   * Creates a new instance of the FullStoryShipper.
   * @param config {@link FullStoryShipperConfig}
   * @param initContext {@link AnalyticsClientInitContext}
   */
  constructor(config, initContext) {
    (0, _defineProperty2.default)(this, "fullStoryApi", void 0);
    (0, _defineProperty2.default)(this, "lastUserId", void 0);
    (0, _defineProperty2.default)(this, "eventTypesAllowlist", void 0);
    (0, _defineProperty2.default)(this, "pageContext$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "userContext$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "subscriptions", new _rxjs.Subscription());
    this.initContext = initContext;
    const {
      eventTypesAllowlist,
      pageVarsDebounceTimeMs = 500,
      ...snippetConfig
    } = config;
    this.fullStoryApi = (0, _load_snippet.loadSnippet)(snippetConfig);
    this.eventTypesAllowlist = eventTypesAllowlist;
    this.subscriptions.add(this.userContext$.pipe((0, _rxjs.distinct)(({
      userId,
      isElasticCloudUser,
      cloudIsElasticStaffOwned,
      cloudTrialEndDate
    }) => [userId, isElasticCloudUser, cloudIsElasticStaffOwned, cloudTrialEndDate].join('-'))).subscribe(userVars => this.updateUserVars(userVars)));
    this.subscriptions.add(this.pageContext$.pipe((0, _rxjs.map)(newContext => {
      // Cherry-picking fields because FS limits the number of fields that can be sent.
      // > Note: You can capture up to 20 unique page properties (exclusive of pageName) for any given page
      // > and up to 500 unique page properties across all pages.
      // https://help.fullstory.com/hc/en-us/articles/1500004101581-FS-setVars-API-Sending-custom-page-data-to-FullStory
      return PAGE_VARS_KEYS.reduce((acc, key) => {
        if ((0, _lodash.has)(newContext, key)) {
          (0, _saferLodashSet.set)(acc, key, (0, _lodash.get)(newContext, key));
        }
        return acc;
      }, {});
    }), (0, _rxjs.filter)(pageVars => Object.keys(pageVars).length > 0),
    // Wait for anything to actually change.
    (0, _rxjs.distinct)(pageVars => {
      const sortedKeys = Object.keys(pageVars).sort();
      return sortedKeys.map(key => pageVars[key]).join('-');
    }),
    // We need some debounce time to ensure everything is updated before calling FS because some properties cannot be changed twice for the same URL.
    (0, _rxjs.debounceTime)(pageVarsDebounceTimeMs)).subscribe(pageVars => {
      this.initContext.logger.debug(`Calling FS.setVars with context ${JSON.stringify(pageVars)}`);
      this.fullStoryApi.setVars('page', {
        ...(0, _format_payload.formatPayload)(pageVars),
        ...(pageVars.version ? (0, _get_parsed_version.getParsedVersion)(pageVars.version) : {})
      });
    }));
  }

  /**
   * Calls `fs.identify`, `fs.setUserVars` and `fs.setVars` depending on the fields provided in the newContext.
   * @param newContext The full new context to set {@link EventContext}
   */
  extendContext(newContext) {
    this.initContext.logger.debug(`Received context ${JSON.stringify(newContext)}`);

    // FullStory requires different APIs for different type of contexts:
    // User-level context.
    this.userContext$.next(newContext);
    // Event-level context. At the moment, only the scope `page` is supported by FullStory for webapps.
    this.pageContext$.next(newContext);
  }

  /**
   * Stops/restarts the shipping mechanism based on the value of isOptedIn
   * @param isOptedIn `true` for resume sending events. `false` to stop.
   */
  optIn(isOptedIn) {
    this.initContext.logger.debug(`Setting FS to optIn ${isOptedIn}`);
    // FullStory uses 2 different opt-in methods:
    // - `consent` is needed to allow collecting information about the components
    //   declared as "Record with user consent" (https://help.fullstory.com/hc/en-us/articles/360020623574).
    //   We need to explicitly call `consent` if for the "Record with user content" feature to work.
    this.fullStoryApi.consent(isOptedIn);
    // - `restart` and `shutdown` fully start/stop the collection of data.
    if (isOptedIn) {
      this.fullStoryApi.restart();
    } else {
      this.fullStoryApi.shutdown();
    }
  }

  /**
   * Filters the events by the eventTypesAllowlist from the config.
   * Then it transforms the event into a FS valid format and calls `fs.event`.
   * @param events batched events {@link Event}
   */
  reportEvents(events) {
    this.initContext.logger.debug(`Reporting ${events.length} events to FS`);
    events.filter(event => {
      var _this$eventTypesAllow, _this$eventTypesAllow2;
      return (_this$eventTypesAllow = (_this$eventTypesAllow2 = this.eventTypesAllowlist) === null || _this$eventTypesAllow2 === void 0 ? void 0 : _this$eventTypesAllow2.includes(event.event_type)) !== null && _this$eventTypesAllow !== void 0 ? _this$eventTypesAllow : true;
    }).forEach(event => {
      // We only read event.properties and discard the rest because the context is already sent in the other APIs.
      this.fullStoryApi.event(event.event_type, (0, _format_payload.formatPayload)(event.properties));
    });
  }

  /**
   * Flushes all internal queues of the shipper.
   * It doesn't really do anything inside because this shipper doesn't hold any internal queues.
   */
  async flush() {}

  /**
   * Shuts down the shipper.
   */
  shutdown() {
    this.subscriptions.unsubscribe();
  }
  updateUserVars({
    userId,
    isElasticCloudUser,
    cloudIsElasticStaffOwned,
    cloudTrialEndDate
  }) {
    // Call it only when the userId changes
    if (userId && userId !== this.lastUserId) {
      this.initContext.logger.debug(`Calling FS.identify with userId ${userId}`);
      // We need to call the API for every new userId (restarting the session).
      this.fullStoryApi.identify(userId);
      this.lastUserId = userId;
    }

    // User-level context
    if (typeof isElasticCloudUser === 'boolean' || typeof cloudIsElasticStaffOwned === 'boolean' || cloudTrialEndDate) {
      const userVars = {
        isElasticCloudUser,
        cloudIsElasticStaffOwned,
        cloudTrialEndDate
      };
      this.initContext.logger.debug(`Calling FS.setUserVars with ${JSON.stringify(userVars)}`);
      this.fullStoryApi.setUserVars((0, _format_payload.formatPayload)(userVars));
    }
  }
}
exports.FullStoryShipper = FullStoryShipper;
/** Shipper's unique name */
(0, _defineProperty2.default)(FullStoryShipper, "shipperName", 'FullStory');