"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticV3BrowserShipper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Elastic V3 shipper to use in the browser.
 */
class ElasticV3BrowserShipper {
  /**
   * Creates a new instance of the {@link ElasticV3BrowserShipper}.
   * @param options {@link ElasticV3ShipperOptions}
   * @param initContext {@link AnalyticsClientInitContext}
   */
  constructor(options, initContext) {
    var _options$sendTo;
    /** Observable to emit the stats of the processed events. */
    (0, _defineProperty2.default)(this, "telemetryCounter$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "reportTelemetryCounters", (0, _common.createTelemetryCounterHelper)(this.telemetryCounter$, ElasticV3BrowserShipper.shipperName));
    (0, _defineProperty2.default)(this, "url", void 0);
    (0, _defineProperty2.default)(this, "internalQueue$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "flush$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "queueFlushed$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "isOptedIn$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "clusterUuid", 'UNKNOWN');
    (0, _defineProperty2.default)(this, "licenseId", void 0);
    this.options = options;
    this.initContext = initContext;
    this.setUpInternalQueueSubscriber();
    this.url = (0, _common.buildUrl)({
      sendTo: (_options$sendTo = options.sendTo) !== null && _options$sendTo !== void 0 ? _options$sendTo : initContext.sendTo,
      channelName: options.channelName
    });
  }

  /**
   * Uses the `cluster_uuid` and `license_id` from the context to hold them in memory for the generation of the headers
   * used later on in the HTTP request.
   * @param newContext The full new context to set {@link EventContext}
   */
  extendContext(newContext) {
    if (newContext.cluster_uuid) {
      this.clusterUuid = newContext.cluster_uuid;
    }
    if (newContext.license_id) {
      this.licenseId = newContext.license_id;
    }
  }

  /**
   * When `false`, it flushes the internal queue and stops sending events.
   * @param isOptedIn `true` for resume sending events. `false` to stop.
   */
  optIn(isOptedIn) {
    this.isOptedIn$.next(isOptedIn);
  }

  /**
   * Enqueues the events to be sent to in a batched approach.
   * @param events batched events {@link Event}
   */
  reportEvents(events) {
    events.forEach(event => {
      this.internalQueue$.next(event);
    });
  }

  /**
   * Triggers a flush of the internal queue to attempt to send any events held in the queue
   * and resolves the returned promise once the queue is emptied.
   */
  async flush() {
    if (this.flush$.isStopped) {
      // If called after shutdown, return straight away
      return;
    }
    const promise = (0, _rxjs.firstValueFrom)(this.queueFlushed$);
    this.flush$.next();
    await promise;
  }

  /**
   * Shuts down the shipper.
   * Triggers a flush of the internal queue to attempt to send any events held in the queue.
   */
  shutdown() {
    this.internalQueue$.complete(); // NOTE: When completing the observable, the buffer logic does not wait and releases any buffered events.
    this.flush$.complete();
  }
  setUpInternalQueueSubscriber() {
    this.internalQueue$.pipe(
    // Buffer events for 1 second or until we have an optIn value
    (0, _rxjs.bufferWhen)(() => (0, _rxjs.merge)(this.flush$, (0, _rxjs.interval)(1000).pipe((0, _rxjs.skipWhile)(() => this.isOptedIn$.value === undefined)))),
    // Send events (one batch at a time)
    (0, _rxjs.concatMap)(async events => {
      // Only send if opted-in and there's anything to send
      if (this.isOptedIn$.value === true && events.length > 0) {
        await this.sendEvents(events);
      }
    }), (0, _rxjs.map)(() => this.queueFlushed$.next())).subscribe();
  }
  async sendEvents(events) {
    try {
      const code = await this.makeRequest(events);
      this.reportTelemetryCounters(events, {
        code
      });
    } catch (error) {
      this.reportTelemetryCounters(events, {
        code: error.code,
        error
      });
    }
  }
  async makeRequest(events) {
    const response = await fetch(this.url, {
      method: 'POST',
      body: (0, _common.eventsToNDJSON)(events),
      headers: (0, _common.buildHeaders)(this.clusterUuid, this.options.version, this.licenseId),
      ...(this.options.debug && {
        query: {
          debug: true
        }
      }),
      // Allow the request to outlive the page in case the tab is closed
      keepalive: true
    });
    if (this.options.debug) {
      this.initContext.logger.debug(`[${ElasticV3BrowserShipper.shipperName}]: ${response.status} - ${await response.text()}`);
    }
    if (!response.ok) {
      throw new _common.ErrorWithCode(`${response.status} - ${await response.text()}`, `${response.status}`);
    }
    return `${response.status}`;
  }
}
exports.ElasticV3BrowserShipper = ElasticV3BrowserShipper;
/** Shipper's unique name */
(0, _defineProperty2.default)(ElasticV3BrowserShipper, "shipperName", 'elastic_v3_browser');