"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEsqlMode = useEsqlMode;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _react = require("react");
var _rxjs = require("rxjs");
var _discover_state_provider = require("../state_management/discover_state_provider");
var _get_valid_view_mode = require("../utils/get_valid_view_mode");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_NUM_OF_COLUMNS = 50;
/**
 * Hook to take care of ES|QL state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
function useEsqlMode({
  dataViews,
  stateContainer
}) {
  const prev = (0, _react.useRef)({
    recentlyUpdatedToColumns: [],
    query: ''
  });
  const initialFetch = (0, _react.useRef)(true);
  const savedSearch = (0, _discover_state_provider.useSavedSearchInitial)();
  const cleanup = (0, _react.useCallback)(() => {
    if (prev.current.query) {
      // cleanup when it's not an ES|QL query
      prev.current = {
        recentlyUpdatedToColumns: [],
        query: ''
      };
      initialFetch.current = true;
    }
  }, []);
  (0, _react.useEffect)(() => {
    const subscription = stateContainer.dataState.data$.documents$.pipe((0, _rxjs.switchMap)(async next => {
      const {
        query
      } = next;
      if (!query || next.fetchStatus === _types.FetchStatus.ERROR) {
        return;
      }
      const sendComplete = () => {
        stateContainer.dataState.data$.documents$.next({
          ...next,
          fetchStatus: _types.FetchStatus.COMPLETE
        });
      };
      const {
        viewMode
      } = stateContainer.appState.getState();
      const isEsqlQuery = (0, _esQuery.isOfAggregateQueryType)(query);
      if (isEsqlQuery) {
        var _next$result;
        const hasResults = Boolean((_next$result = next.result) === null || _next$result === void 0 ? void 0 : _next$result.length);
        if (next.fetchStatus !== _types.FetchStatus.PARTIAL) {
          return;
        }
        let nextColumns = prev.current.recentlyUpdatedToColumns;
        if (hasResults) {
          const firstRow = next.result[0];
          const firstRowColumns = Object.keys(firstRow.raw);
          if ((0, _esqlUtils.hasTransformationalCommand)(query.esql)) {
            nextColumns = firstRowColumns.slice(0, MAX_NUM_OF_COLUMNS);
          } else {
            nextColumns = [];
          }
        }
        if (initialFetch.current) {
          initialFetch.current = false;
          prev.current.query = query.esql;
          prev.current.recentlyUpdatedToColumns = nextColumns;
        }
        const indexPatternChanged = (0, _esqlUtils.getIndexPatternFromESQLQuery)(query.esql) !== (0, _esqlUtils.getIndexPatternFromESQLQuery)(prev.current.query);
        const addColumnsToState = indexPatternChanged || !(0, _lodash.isEqual)(nextColumns, prev.current.recentlyUpdatedToColumns);
        const changeViewMode = viewMode !== (0, _get_valid_view_mode.getValidViewMode)({
          viewMode,
          isEsqlMode: true
        });
        if (!indexPatternChanged && !addColumnsToState && !changeViewMode) {
          sendComplete();
          return;
        }
        prev.current.query = query.esql;
        prev.current.recentlyUpdatedToColumns = nextColumns;

        // just change URL state if necessary
        if (addColumnsToState || changeViewMode) {
          const nextState = {
            ...(addColumnsToState && {
              columns: nextColumns
            }),
            ...(changeViewMode && {
              viewMode: undefined
            })
          };
          await stateContainer.appState.replaceUrlState(nextState);
        }
        sendComplete();
      } else {
        // cleanup for a "regular" query
        cleanup();
      }
    })).subscribe();
    return () => {
      // cleanup for e.g. when savedSearch is switched
      cleanup();
      subscription.unsubscribe();
    };
  }, [dataViews, stateContainer, savedSearch, cleanup]);
}