"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDegradedDocsPaginated = getDegradedDocsPaginated;
var _server = require("@kbn/observability-plugin/server");
var _constants = require("../../../common/constants");
var _es_fields = require("../../../common/es_fields");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SIZE_LIMIT = 10000;
async function getDegradedDocsPaginated(options) {
  var _degradedDocsResponse, _degradedDocsResponse2, _totalDocsResponse$ag, _totalDocsResponse$ag2, _totalDocsResponse$ag3, _totalDocsResponse$ag4;
  const {
    esClient,
    type = _constants.DEFAULT_DATASET_TYPE,
    datasetQuery,
    start,
    end,
    after,
    prevResults = {
      degradedDocs: [],
      docsCount: []
    }
  } = options;
  const datasetQualityESClient = (0, _utils.createDatasetQualityESClient)(esClient);
  const datasetFilter = {
    ...(datasetQuery ? {
      should: [...(0, _utils.wildcardQuery)(_es_fields.DATA_STREAM_DATASET, datasetQuery), ...(0, _utils.wildcardQuery)(_es_fields.DATA_STREAM_NAMESPACE, datasetQuery)],
      minimum_should_match: 1
    } : {})
  };
  const otherFilters = [...(0, _server.rangeQuery)(start, end), ...(0, _server.termQuery)(_es_fields.DATA_STREAM_TYPE, type)];
  const aggs = afterKey => ({
    datasets: {
      composite: {
        ...(afterKey ? {
          after: afterKey
        } : {}),
        size: SIZE_LIMIT,
        sources: [{
          dataset: {
            terms: {
              field: 'data_stream.dataset'
            }
          }
        }, {
          namespace: {
            terms: {
              field: 'data_stream.namespace'
            }
          }
        }]
      }
    }
  });
  const response = await datasetQualityESClient.msearch({
    index: `${type}-*-*`
  }, [
  // degraded docs per dataset
  {
    size: 0,
    query: {
      bool: {
        ...datasetFilter,
        filter: otherFilters,
        must: {
          exists: {
            field: _es_fields._IGNORED
          }
        }
      }
    },
    aggs: aggs(after === null || after === void 0 ? void 0 : after.degradedDocs)
  },
  // total docs per dataset
  {
    size: 0,
    query: {
      bool: {
        ...datasetFilter,
        filter: otherFilters
      }
    },
    aggs: aggs(after === null || after === void 0 ? void 0 : after.docsCount)
  }]);
  const [degradedDocsResponse, totalDocsResponse] = response.responses;
  const currDegradedDocs = (_degradedDocsResponse = (_degradedDocsResponse2 = degradedDocsResponse.aggregations) === null || _degradedDocsResponse2 === void 0 ? void 0 : _degradedDocsResponse2.datasets.buckets.map(bucket => ({
    dataset: `${type}-${bucket.key.dataset}-${bucket.key.namespace}`,
    count: bucket.doc_count
  }))) !== null && _degradedDocsResponse !== void 0 ? _degradedDocsResponse : [];
  const degradedDocs = [...prevResults.degradedDocs, ...currDegradedDocs];
  const currTotalDocs = (_totalDocsResponse$ag = (_totalDocsResponse$ag2 = totalDocsResponse.aggregations) === null || _totalDocsResponse$ag2 === void 0 ? void 0 : _totalDocsResponse$ag2.datasets.buckets.map(bucket => ({
    dataset: `${type}-${bucket.key.dataset}-${bucket.key.namespace}`,
    count: bucket.doc_count
  }))) !== null && _totalDocsResponse$ag !== void 0 ? _totalDocsResponse$ag : [];
  const docsCount = [...prevResults.docsCount, ...currTotalDocs];
  if ((_totalDocsResponse$ag3 = totalDocsResponse.aggregations) !== null && _totalDocsResponse$ag3 !== void 0 && _totalDocsResponse$ag3.datasets.after_key && ((_totalDocsResponse$ag4 = totalDocsResponse.aggregations) === null || _totalDocsResponse$ag4 === void 0 ? void 0 : _totalDocsResponse$ag4.datasets.buckets.length) === SIZE_LIMIT) {
    var _degradedDocsResponse3, _totalDocsResponse$ag5;
    return getDegradedDocsPaginated({
      esClient,
      type,
      start,
      end,
      datasetQuery,
      after: {
        degradedDocs: ((_degradedDocsResponse3 = degradedDocsResponse.aggregations) === null || _degradedDocsResponse3 === void 0 ? void 0 : _degradedDocsResponse3.datasets.after_key) || (after === null || after === void 0 ? void 0 : after.degradedDocs),
        docsCount: ((_totalDocsResponse$ag5 = totalDocsResponse.aggregations) === null || _totalDocsResponse$ag5 === void 0 ? void 0 : _totalDocsResponse$ag5.datasets.after_key) || (after === null || after === void 0 ? void 0 : after.docsCount)
      },
      prevResults: {
        degradedDocs,
        docsCount
      }
    });
  }
  const degradedDocsMap = degradedDocs.reduce((acc, curr) => ({
    ...acc,
    [curr.dataset]: curr.count
  }), {});
  return docsCount.map(curr => {
    const degradedDocsCount = degradedDocsMap[curr.dataset] || 0;
    return {
      ...curr,
      docsCount: curr.count,
      count: degradedDocsCount,
      percentage: degradedDocsCount / curr.count * 100
    };
  });
}