"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPureDatasetQualityControllerStateMachine = exports.createDatasetQualityControllerStateMachine = void 0;
var _timerange = require("@kbn/timerange");
var _xstate = require("xstate");
var _utils = require("../../../../common/utils");
var _utils2 = require("../../../utils");
var _defaults = require("./defaults");
var _notifications = require("./notifications");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPureDatasetQualityControllerStateMachine = initialContext => /** @xstate-layout N4IgpgJg5mDOIC5QBECGAXVszoIoFdUAbAS3QE8BhAewDt0AnaoosBgOggyx1gGIAqgAVkAQQAqAUQD64gJIBZGQCVRAOQDikgNoAGALqJQAB2qwyJOkZAAPRAFZ77AEwAWAGwAOXfc+f39u7uAIwBADQg5IieAMwAnOwA7PbOujHucTHpzs72AL55EWiY2HiEpBQ09EwsbJzcpfzKkgBizQDKABLSYuKieoZIIKbm6Ja01nYIzu4xLv4ZMcGJwZ7OiXERUQieronsurtxibquufYrMQVFDTgExGRUdIzMrBxcJbzsAGY4AMYACxItCgfAgdDA7GBADdqABrSHFHhlB6VZ41N71T7oWA-f5AkEIGHUP4YcYDAbWEYWKxDKbpJwxNJZHIxHK6DKJLaIYI+fauTzBeyuOLOOKuGKxK6FEBI0r3CpPaqvOofZG437oQHA0FsJgcYxEDDfagMAC27Dld3KjyqL1q71uOLxWoJUCJtFhpLGdApBipZhpEzpiD2nhccVWx2CwVch1W9m5Owy7HciT89klIr2rnyMqtKMVdoxqrAUAYqAgkGQJP4wl6MnkSmkqk0On9Q2pPuDoCm3nc7F26aynLOIqTsQSyVS6Uy2Tc11lToVtvRKveZYrVYgNb+TVaHW6vX6HZMge7kx5umWB3WYaynkSMWFE-czlTrj2s2C6yf9mCi4FiuaLKg6nCbpW1a1i62ogmCEJQp68KIsuNogfamJVuWkE7tBmqwe6xLeuSBiUp257jJeCAxiE7AeOmJzPokzF7EmvIXHRgrCqK4qSlkgGoaiSoYaW2HbruGr4jqfB6qa7CGsapoWkBaHCSWG5iVBe4wW6HpemSvqkaewwUbSvY8ss+zXnEsShB49iRlykSIHEhzzMxKQ-s4ni+HmNzYsBanroh6AQd2dYiBIjaKCo6haGRZ6jJRIYIMKqarMsbIhB5QqJs5CCfq4g4xs+nkhDEEruAJAWqcWwXAqF2HhXwzRtJIXQ9BIJ6DIlQZUTkRWZisiQij5qzuBySbOEK7AXCNwTHLxqQpNVyKBXVYENWF4wRQ20hyGoUgaKo8gAPJqO0CUmUlZm2A4HGxJ+Sz+CsDl5dsoRON4mRxvYPgcuKq3yrVa6bfQ210LtUXSGoohKO0QiiJQ7VXV2yXmalngJEyMTDXEKQCksU1pi4cZY2moRzq4QPWkJG2YltTU7YIkVSNIuACJIygAJqo6ZPZ3dRqxFRkszJFluPLGxGzsDZv7pCscbrH5S41XToMM+DTOQzp0ngrQkLEgilqCUWGt1IzFbhbrhJEQZtB+j1119Slua6IOzE+VmPhvm4Sa44yey6AxyzTb4NOFquoGa41Vs7TbuoMPq8lGugJrmibatm9HFta3HOv4bpdvdo7AY3QL9K8rNcTuAK3ixgKyRJoE7t7AHkZxKKDlVfmptRyJHCW-buJENQOEs3tfQAEIADIyJQyhyFIi-dWXLsY2m4a8Y+-7LJvSYTYNWQ1xkFyPjMEfrebg958P7Cj+P4inRoGhz-tsOUPIABqMjHu0kjiEusZNGt0+wzHYBlUUWNj4XFcEmCU+w2Q-lnP4T8mRL4gxzjfWOd8H7bj4E-F+b8WgCBnjPTqfR-7iBhnDFGwD+ZUUFPsea14BS6EyFGf26ROKhw5FjDwAoVYqXVlgn4RByDUHwOgPg-856fxhpIAA6hQ0QVC+blyoimdhvgTiZmcFkBBbEKpFSfN5ZIwQ2S42pr3LO-d1JiIkVIvglAZ6nX-tIFoM9uanQEOIdR69Bb-icJ+RwARYysifEY0IsssZ+A2KcCqmYMEiIHg4yR6BEIWAeAALz1ghWAmBQqZzWpg1J3xxHpMyWMHJOp-EXhSryGacZEj6O8p3XGxw2L4zmMkGywpprDXWMk7OZSKlSKqSQGpIIsSoHaIwMAqAzTIBwKgEgRBJKujyQbRCsJjbCJGfY8pjiMnAiyaQXJ0y1RzIYAspZKy1kbIInpEk9tS7kQ0a7fRqYVhYwWv+XMViuluHYLjYx6xBTsk8MMuxwUjmVNOdU85OoZnXNucszADyE4ySTnJBSaclLFOBikw5YyTm0DOSQC5UAUXzMWei1Z6yE7POIoZfQdT0aCzOMEJI2jryxiCaKJyH10xTl4r4fwPhhSJGhehElxyJlTOpUPbsaBYAAgAEZjwYBAR5bp4LbKNihWxsrYWkoVUi6Zyrxiqo1VqnVTLi4kTZfQj5GM0EuCWN5NY1k1h+3yjkRwEDQ4BAmiOR8MqgpgTheMhFkyLVKtviqrAtrUDat1dJWSBpU7p2Un3E1UazWxsVSFCGtAbWatTfawuOpmWvKMk7EBFdQwig9T+PwqQFq+rgf60I74rEXF0OsTIpwI30zqNGjJeDIDONce4zx3jfHstAYgEaCQT6jWWjmZwbE9hODFKHY4iwghQpsSU4lpr5VTogBPaGB12hyA0J0QBsgYotjiu2BtDCGmdoOEybR4TDjMTYrA1MQRUjBxrrkLwo7r5pPGX8Ue2Br2nSEJINQHivE+L8S6gJUxwUQIqsxNMMxEi13SDumuBx-D40fKcE4J6ZS0GoFWeAQx9kwodGvepGMAC0rhuWSlWCRyUA1djvUQDxre00cwZE-LoYO-gYOiLVI0LjHKpj7DcF4HwfgQ0hHCPlacIKFhiiyJKR81j-JnoOcFFTXxq0gjU8uoW4Z5O7DSIKSUswBRJnFAOLIvhPwzElhkJTqS7POivU5ptqVwFCksSJiU-Hgi+Z8rNGYf0mR-XWOkML9isJbi0qx3q3HBbJHDEOKUo43CbEMykYzYG5zmZGnl2zEFxJ4Sko595uHoiRlmvxxwrkuInBS-lY4744n+FCN4Ea6ZWtgQKzhCS98x7bmi1RCxaW1h6Nco4CqaQ2LXjmMRxw15DhsmFAtmOpbivO1K1MAcIQmFZUskE8TCB0x0QlJkMUPgDFLGu7nHB1sHNQA2w08USRYi128Bmc+7gppQ8Ysgi434fJCLzZGm72taAjzW5ACHG8zgQN8KESUwdgtCocDkA4AQRTsLo14OIQOOATqJ2Vun9FZwnDTHNtiOQEhvk-N4JkHheTOFZ3BslFKqUc6mIGjk3POQcg8mxEITgOQVUFIzrMlnVbWY45iCd5rKXIqubSu5GL1ny6vHMJYsxIyEfxtNRH+VliDrol63TrkYz4ylybot8aaU3LpfcxlYPbfUX-CC8nTu9gu9CGxXYRVTiOVmFxbykvT1EpswW+VQezeXO4KisP1vcT6zAFH0+ECctCkjOcIIyeHLfN5LsGM00KrZ6s7no347C3ksRUXhNIPrXJorWmqPvI5hK7TDz1XTd-XB01+w1YLT4isIDwP2XyKrV0HLXa9N3WSvqZ5DH2fT4Vd84+wMkxPFO4zjTD3HvtM8-G+30PqlJbccH8rRXiEUepG7sYGmYFiwoSCMQU0C0rcsYrkiwZwawW+l6BOEAU+wc3KHCjEaOYoQG7ugQCQ-gHawW-GsQ3eBuve+a7+8qCGZghOPWD290A44oLSg66+2YkB7uEo742iTIywvgg6K0BQeQQAA */
(0, _xstate.createMachine)({
  context: initialContext,
  predictableActionArguments: true,
  id: 'DatasetQualityController',
  type: 'parallel',
  states: {
    datasets: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadDataStreamStats',
            onDone: {
              target: 'loaded',
              actions: ['storeDataStreamStats', 'storeDatasets']
            },
            onError: {
              target: 'loaded',
              actions: ['notifyFetchDatasetStatsFailed']
            }
          }
        },
        loaded: {}
      },
      on: {
        UPDATE_TIME_RANGE: {
          target: 'datasets.fetching',
          actions: ['storeTimeRange']
        },
        REFRESH_DATA: {
          target: 'datasets.fetching'
        }
      }
    },
    degradedDocs: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadDegradedDocs',
            onDone: {
              target: 'loaded',
              actions: ['storeDegradedDocStats', 'storeDatasets']
            },
            onError: [{
              target: 'unauthorized',
              cond: 'checkIfActionForbidden'
            }, {
              target: 'loaded',
              actions: ['notifyFetchDegradedStatsFailed']
            }]
          }
        },
        loaded: {},
        unauthorized: {
          type: 'final'
        }
      },
      on: {
        UPDATE_TIME_RANGE: {
          target: 'degradedDocs.fetching',
          actions: ['storeTimeRange']
        },
        REFRESH_DATA: {
          target: 'degradedDocs.fetching'
        }
      }
    },
    integrations: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadIntegrations',
            onDone: {
              target: 'loaded',
              actions: ['storeIntegrations', 'storeDatasets']
            },
            onError: {
              target: 'loaded',
              actions: ['notifyFetchIntegrationsFailed', 'storeEmptyIntegrations', 'storeDatasets']
            }
          }
        },
        loaded: {
          on: {
            UPDATE_TABLE_CRITERIA: {
              target: 'loaded',
              actions: ['storeTableOptions']
            },
            TOGGLE_INACTIVE_DATASETS: {
              target: 'loaded',
              actions: ['storeInactiveDatasetsVisibility', 'resetPage']
            },
            TOGGLE_FULL_DATASET_NAMES: {
              target: 'loaded',
              actions: ['storeFullDatasetNamesVisibility']
            }
          }
        }
      },
      on: {
        UPDATE_TIME_RANGE: {
          target: 'integrations.fetching',
          actions: ['storeTimeRange']
        },
        REFRESH_DATA: {
          target: 'integrations.fetching'
        },
        UPDATE_INTEGRATIONS: {
          target: 'integrations.loaded',
          actions: ['storeIntegrationsFilter']
        },
        UPDATE_NAMESPACES: {
          target: 'integrations.loaded',
          actions: ['storeNamespaces']
        },
        UPDATE_QUALITIES: {
          target: 'integrations.loaded',
          actions: ['storeQualities']
        },
        UPDATE_QUERY: {
          actions: ['storeQuery']
        }
      }
    },
    nonAggregatableDatasets: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadNonAggregatableDatasets',
            onDone: {
              target: 'loaded',
              actions: ['storeNonAggregatableDatasets']
            },
            onError: [{
              target: 'unauthorized',
              cond: 'checkIfActionForbidden'
            }, {
              target: 'loaded',
              actions: ['notifyFetchNonAggregatableDatasetsFailed']
            }]
          }
        },
        loaded: {},
        unauthorized: {
          type: 'final'
        }
      },
      on: {
        UPDATE_TIME_RANGE: {
          target: 'nonAggregatableDatasets.fetching'
        },
        REFRESH_DATA: {
          target: 'nonAggregatableDatasets.fetching'
        }
      }
    },
    flyout: {
      initial: 'closed',
      states: {
        initializing: {
          type: 'parallel',
          states: {
            nonAggregatableDataset: {
              initial: 'fetching',
              states: {
                fetching: {
                  invoke: {
                    src: 'loadDatasetIsNonAggregatable',
                    onDone: {
                      target: 'done',
                      actions: ['storeDatasetIsNonAggregatable']
                    },
                    onError: {
                      target: 'done',
                      actions: ['notifyFetchNonAggregatableDatasetsFailed']
                    }
                  }
                },
                done: {
                  on: {
                    UPDATE_INSIGHTS_TIME_RANGE: {
                      target: 'fetching',
                      actions: ['storeFlyoutOptions']
                    },
                    SELECT_DATASET: {
                      target: 'fetching',
                      actions: ['storeFlyoutOptions']
                    }
                  }
                }
              }
            },
            dataStreamSettings: {
              initial: 'fetching',
              states: {
                fetching: {
                  invoke: {
                    src: 'loadDataStreamSettings',
                    onDone: {
                      target: 'done',
                      actions: ['storeDataStreamSettings']
                    },
                    onError: {
                      target: 'done',
                      actions: ['notifyFetchDatasetSettingsFailed']
                    }
                  }
                },
                done: {
                  type: 'final'
                }
              }
            },
            dataStreamDetails: {
              initial: 'fetching',
              states: {
                fetching: {
                  invoke: {
                    src: 'loadDataStreamDetails',
                    onDone: {
                      target: 'done',
                      actions: ['storeDatasetDetails']
                    },
                    onError: {
                      target: 'done',
                      actions: ['notifyFetchDatasetDetailsFailed']
                    }
                  }
                },
                done: {
                  on: {
                    UPDATE_INSIGHTS_TIME_RANGE: {
                      target: 'fetching',
                      actions: ['storeFlyoutOptions']
                    },
                    BREAKDOWN_FIELD_CHANGE: {
                      actions: ['storeFlyoutOptions']
                    }
                  }
                }
              }
            },
            integrationDashboards: {
              initial: 'fetching',
              states: {
                fetching: {
                  invoke: {
                    src: 'loadIntegrationDashboards',
                    onDone: {
                      target: 'done',
                      actions: ['storeIntegrationDashboards']
                    },
                    onError: [{
                      target: 'unauthorized',
                      cond: 'checkIfActionForbidden'
                    }, {
                      target: 'done',
                      actions: ['notifyFetchIntegrationDashboardsFailed']
                    }]
                  }
                },
                done: {
                  type: 'final'
                },
                unauthorized: {
                  type: 'final'
                }
              }
            },
            dataStreamDegradedFields: {
              initial: 'fetching',
              states: {
                fetching: {
                  invoke: {
                    src: 'loadDegradedFieldsPerDataStream',
                    onDone: {
                      target: 'done',
                      actions: ['storeDegradedFields']
                    },
                    onError: {
                      target: 'done'
                    }
                  }
                },
                done: {
                  on: {
                    UPDATE_INSIGHTS_TIME_RANGE: {
                      target: 'fetching',
                      actions: ['resetDegradedFieldPage']
                    },
                    UPDATE_DEGRADED_FIELDS_TABLE_CRITERIA: {
                      target: 'done',
                      actions: ['storeDegradedFieldTableOptions']
                    }
                  }
                }
              }
            }
          },
          onDone: {
            target: '#DatasetQualityController.flyout.loaded'
          }
        },
        loaded: {
          on: {
            CLOSE_FLYOUT: {
              target: 'closed',
              actions: ['resetFlyoutOptions']
            }
          }
        },
        closed: {
          on: {
            OPEN_FLYOUT: {
              target: '#DatasetQualityController.flyout.initializing',
              actions: ['storeFlyoutOptions']
            }
          }
        }
      },
      on: {
        SELECT_NEW_DATASET: {
          target: '#DatasetQualityController.flyout.initializing',
          actions: ['storeFlyoutOptions']
        },
        CLOSE_FLYOUT: {
          target: '#DatasetQualityController.flyout.closed',
          actions: ['resetFlyoutOptions']
        }
      }
    }
  }
}, {
  actions: {
    storeTableOptions: (0, _xstate.assign)((_context, event) => {
      return 'dataset_criteria' in event ? {
        table: event.dataset_criteria
      } : {};
    }),
    storeDegradedFieldTableOptions: (0, _xstate.assign)((context, event) => {
      return 'degraded_field_criteria' in event ? {
        flyout: {
          ...context.flyout,
          degradedFields: {
            ...context.flyout.degradedFields,
            table: event.degraded_field_criteria
          }
        }
      } : {};
    }),
    resetPage: (0, _xstate.assign)((context, _event) => ({
      table: {
        ...context.table,
        page: 0
      }
    })),
    resetDegradedFieldPage: (0, _xstate.assign)((context, _event) => ({
      flyout: {
        ...context.flyout,
        degradedFields: {
          ...context.flyout.degradedFields,
          table: {
            ...context.flyout.degradedFields.table,
            page: 0,
            rowsPerPage: 10
          }
        }
      }
    })),
    storeInactiveDatasetsVisibility: (0, _xstate.assign)((context, _event) => {
      return {
        filters: {
          ...context.filters,
          inactive: !context.filters.inactive
        }
      };
    }),
    storeFullDatasetNamesVisibility: (0, _xstate.assign)((context, _event) => {
      return {
        filters: {
          ...context.filters,
          fullNames: !context.filters.fullNames
        }
      };
    }),
    storeTimeRange: (0, _xstate.assign)((context, event) => {
      return 'timeRange' in event ? {
        filters: {
          ...context.filters,
          timeRange: event.timeRange
        }
      } : {};
    }),
    storeIntegrationsFilter: (0, _xstate.assign)((context, event) => {
      return 'integrations' in event ? {
        filters: {
          ...context.filters,
          integrations: event.integrations
        }
      } : {};
    }),
    storeNamespaces: (0, _xstate.assign)((context, event) => {
      return 'namespaces' in event ? {
        filters: {
          ...context.filters,
          namespaces: event.namespaces
        }
      } : {};
    }),
    storeQualities: (0, _xstate.assign)((context, event) => {
      return 'qualities' in event ? {
        filters: {
          ...context.filters,
          qualities: event.qualities
        }
      } : {};
    }),
    storeQuery: (0, _xstate.assign)((context, event) => {
      return 'query' in event ? {
        filters: {
          ...context.filters,
          query: event.query
        }
      } : {};
    }),
    storeFlyoutOptions: (0, _xstate.assign)((context, event) => {
      var _context$flyout$insig, _context$flyout, _context$filters, _context$flyout2, _event$breakdownField, _context$flyout3;
      const insightsTimeRange = 'timeRange' in event ? event.timeRange : (_context$flyout$insig = (_context$flyout = context.flyout) === null || _context$flyout === void 0 ? void 0 : _context$flyout.insightsTimeRange) !== null && _context$flyout$insig !== void 0 ? _context$flyout$insig : (_context$filters = context.filters) === null || _context$filters === void 0 ? void 0 : _context$filters.timeRange;
      const dataset = 'dataset' in event ? event.dataset : (_context$flyout2 = context.flyout) === null || _context$flyout2 === void 0 ? void 0 : _context$flyout2.dataset;
      const breakdownField = 'breakdownField' in event ? (_event$breakdownField = event.breakdownField) !== null && _event$breakdownField !== void 0 ? _event$breakdownField : undefined : (_context$flyout3 = context.flyout) === null || _context$flyout3 === void 0 ? void 0 : _context$flyout3.breakdownField;
      return {
        flyout: {
          ...context.flyout,
          dataset,
          insightsTimeRange,
          breakdownField
        }
      };
    }),
    resetFlyoutOptions: (0, _xstate.assign)((_context, _event) => ({
      flyout: _defaults.DEFAULT_CONTEXT.flyout
    })),
    storeDataStreamStats: (0, _xstate.assign)((_context, event) => {
      if ('data' in event && 'dataStreamsStats' in event.data) {
        const dataStreamStats = event.data.dataStreamsStats;
        const datasetUserPrivileges = event.data.datasetUserPrivileges;

        // Check if any DataStreamStat has null; to check for serverless
        const isSizeStatsAvailable = !dataStreamStats.length || dataStreamStats.some(stat => stat.totalDocs !== null);
        return {
          dataStreamStats,
          isSizeStatsAvailable,
          datasetUserPrivileges
        };
      }
      return {};
    }),
    storeDegradedDocStats: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        degradedDocStats: event.data
      } : {};
    }),
    storeDegradedFields: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        flyout: {
          ...context.flyout,
          degradedFields: {
            ...context.flyout.degradedFields,
            data: event.data.degradedFields
          }
        }
      } : {};
    }),
    storeNonAggregatableDatasets: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        nonAggregatableDatasets: event.data.datasets
      } : {};
    }),
    storeDataStreamSettings: (0, _xstate.assign)((context, event) => {
      var _event$data;
      return 'data' in event ? {
        flyout: {
          ...context.flyout,
          datasetSettings: (_event$data = event.data) !== null && _event$data !== void 0 ? _event$data : {}
        }
      } : {};
    }),
    storeDatasetDetails: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        flyout: {
          ...context.flyout,
          datasetDetails: event.data
        }
      } : {};
    }),
    storeDatasetIsNonAggregatable: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        flyout: {
          ...context.flyout,
          isNonAggregatable: !event.data.aggregatable
        }
      } : {};
    }),
    storeIntegrations: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        integrations: event.data
      } : {};
    }),
    storeEmptyIntegrations: (0, _xstate.assign)(_context => {
      return {
        integrations: []
      };
    }),
    storeIntegrationDashboards: (0, _xstate.assign)((context, event) => {
      var _context$flyout$datas;
      return 'data' in event && 'dashboards' in event.data ? {
        flyout: {
          ...context.flyout,
          dataset: {
            ...context.flyout.dataset,
            integration: {
              ...((_context$flyout$datas = context.flyout.dataset) === null || _context$flyout$datas === void 0 ? void 0 : _context$flyout$datas.integration),
              dashboards: event.data.dashboards
            }
          }
        }
      } : {};
    }),
    storeDatasets: (0, _xstate.assign)((context, _event) => {
      return context.integrations && (context.dataStreamStats || context.degradedDocStats) ? {
        datasets: (0, _utils2.generateDatasets)(context.dataStreamStats, context.degradedDocStats, context.integrations)
      } : {};
    })
  },
  guards: {
    checkIfActionForbidden: (context, event) => {
      return 'data' in event && 'statusCode' in event.data && event.data.statusCode === 403;
    }
  }
});
exports.createPureDatasetQualityControllerStateMachine = createPureDatasetQualityControllerStateMachine;
const createDatasetQualityControllerStateMachine = ({
  initialContext = _defaults.DEFAULT_CONTEXT,
  toasts,
  dataStreamStatsClient,
  dataStreamDetailsClient
}) => createPureDatasetQualityControllerStateMachine(initialContext).withConfig({
  actions: {
    notifyFetchDatasetStatsFailed: (_context, event) => (0, _notifications.fetchDatasetStatsFailedNotifier)(toasts, event.data),
    notifyFetchDegradedStatsFailed: (_context, event) => (0, _notifications.fetchDegradedStatsFailedNotifier)(toasts, event.data),
    notifyFetchNonAggregatableDatasetsFailed: (_context, event) => (0, _notifications.fetchNonAggregatableDatasetsFailedNotifier)(toasts, event.data),
    notifyFetchDatasetSettingsFailed: (_context, event) => (0, _notifications.fetchDatasetSettingsFailedNotifier)(toasts, event.data),
    notifyFetchDatasetDetailsFailed: (_context, event) => (0, _notifications.fetchDatasetDetailsFailedNotifier)(toasts, event.data),
    notifyFetchIntegrationDashboardsFailed: (_context, event) => (0, _notifications.fetchIntegrationDashboardsFailedNotifier)(toasts, event.data),
    notifyFetchIntegrationsFailed: (_context, event) => (0, _notifications.fetchIntegrationsFailedNotifier)(toasts, event.data)
  },
  services: {
    loadDataStreamStats: context => dataStreamStatsClient.getDataStreamsStats({
      type: context.type,
      datasetQuery: context.filters.query
    }),
    loadDegradedDocs: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.filters.timeRange);
      return dataStreamStatsClient.getDataStreamsDegradedStats({
        type: context.type,
        datasetQuery: context.filters.query,
        start,
        end
      });
    },
    loadDegradedFieldsPerDataStream: context => {
      if (!context.flyout.dataset || !context.flyout.insightsTimeRange) {
        return Promise.resolve({});
      }
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.flyout.insightsTimeRange);
      const {
        type,
        name: dataset,
        namespace
      } = context.flyout.dataset;
      return dataStreamDetailsClient.getDataStreamDegradedFields({
        dataStream: (0, _utils.dataStreamPartsToIndexName)({
          type: type,
          dataset,
          namespace
        }),
        start,
        end
      });
    },
    loadIntegrations: context => {
      return dataStreamStatsClient.getIntegrations({
        type: context.type
      });
    },
    loadNonAggregatableDatasets: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.filters.timeRange);
      return dataStreamStatsClient.getNonAggregatableDatasets({
        type: context.type,
        start,
        end
      });
    },
    loadDataStreamSettings: context => {
      if (!context.flyout.dataset) {
        (0, _notifications.fetchDatasetSettingsFailedNotifier)(toasts, new Error(_notifications.noDatasetSelected));
        return Promise.resolve({});
      }
      const {
        type,
        name: dataset,
        namespace
      } = context.flyout.dataset;
      return dataStreamDetailsClient.getDataStreamSettings({
        dataStream: (0, _utils.dataStreamPartsToIndexName)({
          type: type,
          dataset,
          namespace
        })
      });
    },
    loadDataStreamDetails: context => {
      if (!context.flyout.dataset || !context.flyout.insightsTimeRange) {
        (0, _notifications.fetchDatasetDetailsFailedNotifier)(toasts, new Error(_notifications.noDatasetSelected));
        return Promise.resolve({});
      }
      const {
        type,
        name: dataset,
        namespace
      } = context.flyout.dataset;
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.flyout.insightsTimeRange);
      return dataStreamDetailsClient.getDataStreamDetails({
        dataStream: (0, _utils.dataStreamPartsToIndexName)({
          type: type,
          dataset,
          namespace
        }),
        start,
        end
      });
    },
    loadIntegrationDashboards: context => {
      if (!context.flyout.dataset) {
        (0, _notifications.fetchDatasetDetailsFailedNotifier)(toasts, new Error(_notifications.noDatasetSelected));
        return Promise.resolve({});
      }
      const {
        integration
      } = context.flyout.dataset;
      return integration ? dataStreamDetailsClient.getIntegrationDashboards({
        integration: integration.name
      }) : Promise.resolve({});
    },
    loadDatasetIsNonAggregatable: async context => {
      if (!context.flyout.dataset || !context.flyout.insightsTimeRange) {
        (0, _notifications.fetchDatasetDetailsFailedNotifier)(toasts, new Error(_notifications.noDatasetSelected));
        return Promise.resolve({});
      }
      const {
        type,
        name: dataset,
        namespace
      } = context.flyout.dataset;
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.flyout.insightsTimeRange);
      return dataStreamStatsClient.getNonAggregatableDatasets({
        type: context.type,
        start,
        end,
        dataStream: (0, _utils.dataStreamPartsToIndexName)({
          type: type,
          dataset,
          namespace
        })
      });
    }
  }
});
exports.createDatasetQualityControllerStateMachine = createDatasetQualityControllerStateMachine;