"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRedirectLinkTelemetry = exports.useDatasetTelemetry = exports.useDatasetDetailsTelemetry = void 0;
var _react = require("react");
var _react2 = require("@xstate/react");
var _timerange = require("@kbn/timerange");
var _utils = require("../../common/utils");
var _telemetry = require("../services/telemetry");
var _context = require("../components/dataset_quality/context");
var _use_dataset_quality_filters = require("./use_dataset_quality_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useRedirectLinkTelemetry = ({
  rawName,
  isLogsExplorer,
  telemetry,
  query
}) => {
  const {
    trackDatasetNavigated
  } = useDatasetTelemetry();
  const {
    trackDetailsNavigated,
    navigationTargets
  } = useDatasetDetailsTelemetry();
  const sendTelemetry = (0, _react.useCallback)(() => {
    if (telemetry) {
      const isIgnoredFilter = query ? JSON.stringify(query).includes('_ignored') : false;
      if (telemetry.page === 'main') {
        trackDatasetNavigated(rawName, isIgnoredFilter);
      } else {
        trackDetailsNavigated(isLogsExplorer ? navigationTargets.LogsExplorer : navigationTargets.Discover, telemetry.navigationSource, isIgnoredFilter);
      }
    }
  }, [isLogsExplorer, trackDetailsNavigated, navigationTargets, query, rawName, telemetry, trackDatasetNavigated]);
  const wrapLinkPropsForTelemetry = (0, _react.useCallback)(props => {
    return {
      ...props,
      onClick: event => {
        sendTelemetry();
        if (props.onClick) {
          props.onClick(event);
        }
      }
    };
  }, [sendTelemetry]);
  return {
    wrapLinkPropsForTelemetry,
    sendTelemetry
  };
};
exports.useRedirectLinkTelemetry = useRedirectLinkTelemetry;
const useDatasetTelemetry = () => {
  var _useSelector;
  const {
    service,
    telemetryClient
  } = (0, _context.useDatasetQualityContext)();

  // eslint-disable-next-line react-hooks/exhaustive-deps
  const datasets = (_useSelector = (0, _react2.useSelector)(service, state => state.context.datasets)) !== null && _useSelector !== void 0 ? _useSelector : {};
  const nonAggregatableDatasets = (0, _react2.useSelector)(service, state => state.context.nonAggregatableDatasets);
  const canUserViewIntegrations = (0, _react2.useSelector)(service, state => state.context.datasetUserPrivileges.canViewIntegrations);
  const sort = (0, _react2.useSelector)(service, state => state.context.table.sort);
  const appliedFilters = (0, _use_dataset_quality_filters.useDatasetQualityFilters)();
  const trackDatasetNavigated = (0, _react.useCallback)((rawName, isIgnoredFilter) => {
    const foundDataset = datasets.find(dataset => dataset.rawName === rawName);
    if (foundDataset) {
      const ebtProps = getDatasetEbtProps(foundDataset, sort, appliedFilters, nonAggregatableDatasets, isIgnoredFilter, canUserViewIntegrations);
      telemetryClient.trackDatasetNavigated(ebtProps);
    } else {
      throw new Error(`Cannot report dataset navigation telemetry for unknown dataset ${rawName}`);
    }
  }, [sort, appliedFilters, canUserViewIntegrations, datasets, nonAggregatableDatasets, telemetryClient]);
  return {
    trackDatasetNavigated
  };
};
exports.useDatasetTelemetry = useDatasetTelemetry;
const useDatasetDetailsTelemetry = () => {
  var _useSelector2;
  const {
    service,
    telemetryClient
  } = (0, _context.useDatasetQualityContext)();
  const {
    dataset: dataStreamStat,
    datasetDetails: dataStreamDetails,
    insightsTimeRange,
    breakdownField,
    isNonAggregatable
  } = (_useSelector2 = (0, _react2.useSelector)(service, state => state.context.flyout)) !== null && _useSelector2 !== void 0 ? _useSelector2 : {};
  const loadingState = (0, _react2.useSelector)(service, state => ({
    dataStreamDetailsLoading: state.matches('flyout.initializing.dataStreamDetails.fetching')
  }));
  const canUserAccessDashboards = (0, _react2.useSelector)(service, state => !state.matches('flyout.initializing.integrationDashboards.unauthorized'));
  const canUserViewIntegrations = (0, _react2.useSelector)(service, state => state.context.datasetUserPrivileges.canViewIntegrations);
  const ebtProps = (0, _react.useMemo)(() => {
    if (dataStreamDetails && insightsTimeRange && dataStreamStat && !loadingState.dataStreamDetailsLoading) {
      return getDatasetDetailsEbtProps(insightsTimeRange, dataStreamStat, dataStreamDetails, isNonAggregatable !== null && isNonAggregatable !== void 0 ? isNonAggregatable : false, canUserViewIntegrations, canUserAccessDashboards, breakdownField);
    }
    return undefined;
  }, [insightsTimeRange, dataStreamStat, dataStreamDetails, loadingState.dataStreamDetailsLoading, isNonAggregatable, canUserViewIntegrations, canUserAccessDashboards, breakdownField]);
  const startTracking = (0, _react.useCallback)(() => {
    telemetryClient.startDatasetDetailsTracking();
  }, [telemetryClient]);

  // Report opening dataset details
  (0, _react.useEffect)(() => {
    const datasetDetailsTrackingState = telemetryClient.getDatasetDetailsTrackingState();
    if (datasetDetailsTrackingState === 'started' && ebtProps) {
      telemetryClient.trackDatasetDetailsOpened(ebtProps);
    }
  }, [ebtProps, telemetryClient]);
  const trackDetailsNavigated = (0, _react.useCallback)((target, source, isDegraded = false) => {
    const datasetDetailsTrackingState = telemetryClient.getDatasetDetailsTrackingState();
    if ((datasetDetailsTrackingState === 'opened' || datasetDetailsTrackingState === 'navigated') && ebtProps) {
      telemetryClient.trackDatasetDetailsNavigated({
        ...ebtProps,
        filters: {
          is_degraded: isDegraded
        },
        target,
        source
      });
    } else {
      throw new Error('Cannot report dataset details navigation telemetry without required data and state');
    }
  }, [ebtProps, telemetryClient]);
  const wrapLinkPropsForTelemetry = (0, _react.useCallback)((props, target, source, isDegraded = false) => {
    return {
      ...props,
      onClick: event => {
        trackDetailsNavigated(target, source, isDegraded);
        if (props.onClick) {
          props.onClick(event);
        }
      }
    };
  }, [trackDetailsNavigated]);
  return {
    startTracking,
    trackDetailsNavigated,
    wrapLinkPropsForTelemetry,
    navigationTargets: _telemetry.NavigationTarget,
    navigationSources: _telemetry.NavigationSource
  };
};
exports.useDatasetDetailsTelemetry = useDatasetDetailsTelemetry;
function getDatasetEbtProps(dataset, sort, filters, nonAggregatableDatasets, isIgnoredFilter, canUserViewIntegrations) {
  var _dataset$integration, _dataset$userPrivileg, _dataset$userPrivileg2, _filters$selectedQuer, _filters$selectedQuer2;
  const {
    startDate: from,
    endDate: to
  } = (0, _timerange.getDateISORange)(filters.timeRange);
  const datasetEbtProps = {
    index_name: dataset.rawName,
    data_stream: {
      dataset: dataset.name,
      namespace: dataset.namespace,
      type: dataset.type
    },
    data_stream_health: dataset.degradedDocs.quality,
    data_stream_aggregatable: nonAggregatableDatasets.some(indexName => indexName === dataset.rawName),
    from,
    to,
    degraded_percentage: dataset.degradedDocs.percentage,
    integration: (_dataset$integration = dataset.integration) === null || _dataset$integration === void 0 ? void 0 : _dataset$integration.name,
    privileges: {
      can_monitor_data_stream: (_dataset$userPrivileg = (_dataset$userPrivileg2 = dataset.userPrivileges) === null || _dataset$userPrivileg2 === void 0 ? void 0 : _dataset$userPrivileg2.canMonitor) !== null && _dataset$userPrivileg !== void 0 ? _dataset$userPrivileg : true,
      can_view_integrations: canUserViewIntegrations
    }
  };
  const ebtFilters = {
    is_degraded: isIgnoredFilter,
    query_length: (_filters$selectedQuer = (_filters$selectedQuer2 = filters.selectedQuery) === null || _filters$selectedQuer2 === void 0 ? void 0 : _filters$selectedQuer2.length) !== null && _filters$selectedQuer !== void 0 ? _filters$selectedQuer : 0,
    integrations: {
      total: filters.integrations.filter(item => item.name !== 'none').length,
      included: filters.integrations.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'on').length,
      excluded: filters.integrations.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'off').length
    },
    namespaces: {
      total: filters.namespaces.length,
      included: filters.namespaces.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'on').length,
      excluded: filters.namespaces.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'off').length
    },
    qualities: {
      total: filters.qualities.length,
      included: filters.qualities.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'on').length,
      excluded: filters.qualities.filter(item => (item === null || item === void 0 ? void 0 : item.checked) === 'off').length
    }
  };
  return {
    ...datasetEbtProps,
    sort,
    filters: ebtFilters
  };
}
function getDatasetDetailsEbtProps(insightsTimeRange, flyoutDataset, details, isNonAggregatable, canUserViewIntegrations, canUserAccessDashboards, breakdownField) {
  var _details$degradedDocs, _details$docsCount, _flyoutDataset$integr;
  const indexName = flyoutDataset.rawName;
  const dataStream = {
    dataset: flyoutDataset.name,
    namespace: flyoutDataset.namespace,
    type: flyoutDataset.type
  };
  const degradedDocs = (_details$degradedDocs = details === null || details === void 0 ? void 0 : details.degradedDocsCount) !== null && _details$degradedDocs !== void 0 ? _details$degradedDocs : 0;
  const totalDocs = (_details$docsCount = details === null || details === void 0 ? void 0 : details.docsCount) !== null && _details$docsCount !== void 0 ? _details$docsCount : 0;
  const degradedPercentage = totalDocs > 0 ? Number((degradedDocs / totalDocs * 100).toFixed(2)) : 0;
  const health = (0, _utils.mapPercentageToQuality)(degradedPercentage);
  const {
    startDate: from,
    endDate: to
  } = (0, _timerange.getDateISORange)(insightsTimeRange);
  return {
    index_name: indexName,
    data_stream: dataStream,
    privileges: {
      can_monitor_data_stream: true,
      can_view_integrations: canUserViewIntegrations,
      can_view_dashboards: canUserAccessDashboards
    },
    data_stream_aggregatable: !isNonAggregatable,
    data_stream_health: health,
    from,
    to,
    degraded_percentage: degradedPercentage,
    integration: (_flyoutDataset$integr = flyoutDataset.integration) === null || _flyoutDataset$integr === void 0 ? void 0 : _flyoutDataset$integr.name,
    breakdown_field: breakdownField
  };
}