"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRedirectLink = void 0;
var _react = require("react");
var _deeplinksObservability = require("@kbn/deeplinks-observability");
var _common = require("@kbn/discover-plugin/common");
var _esQuery = require("@kbn/es-query");
var _routerUtils = require("@kbn/router-utils");
var _react2 = require("@xstate/react");
var _context = require("../components/dataset_quality/context");
var _utils = require("../utils");
var _use_telemetry = require("./use_telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useRedirectLink = ({
  dataStreamStat,
  query,
  timeRangeConfig,
  breakdownField,
  telemetry
}) => {
  const {
    services: {
      share
    }
  } = (0, _utils.useKibanaContextForPlugin)();
  const {
    service
  } = (0, _context.useDatasetQualityContext)();
  const {
    timeRange
  } = (0, _react2.useSelector)(service, state => state.context.filters);
  const {
    from,
    to
  } = timeRangeConfig || timeRange;
  const logsExplorerLocator = share.url.locators.get(_deeplinksObservability.SINGLE_DATASET_LOCATOR_ID);
  const {
    sendTelemetry
  } = (0, _use_telemetry.useRedirectLinkTelemetry)({
    rawName: dataStreamStat.rawName,
    isLogsExplorer: !!logsExplorerLocator,
    telemetry,
    query
  });
  return (0, _react.useMemo)(() => {
    const config = logsExplorerLocator ? buildLogsExplorerConfig({
      locator: logsExplorerLocator,
      dataStreamStat,
      query,
      from,
      to,
      breakdownField
    }) : buildDiscoverConfig({
      locatorClient: share.url.locators,
      dataStreamStat,
      query,
      from,
      to,
      breakdownField
    });
    const onClickWithTelemetry = event => {
      sendTelemetry();
      if (config.routerLinkProps.onClick) {
        config.routerLinkProps.onClick(event);
      }
    };
    const navigateWithTelemetry = () => {
      sendTelemetry();
      config.navigate();
    };
    return {
      linkProps: {
        ...config.routerLinkProps,
        onClick: onClickWithTelemetry
      },
      navigate: navigateWithTelemetry,
      isLogsExplorerAvailable: !!logsExplorerLocator
    };
  }, [breakdownField, dataStreamStat, from, to, logsExplorerLocator, query, sendTelemetry, share.url.locators]);
};
exports.useRedirectLink = useRedirectLink;
const buildLogsExplorerConfig = ({
  locator,
  dataStreamStat,
  query,
  from,
  to,
  breakdownField
}) => {
  var _dataStreamStat$integ;
  const params = {
    dataset: dataStreamStat.name,
    timeRange: {
      from,
      to
    },
    integration: (_dataStreamStat$integ = dataStreamStat.integration) === null || _dataStreamStat$integ === void 0 ? void 0 : _dataStreamStat$integ.name,
    query,
    filterControls: {
      namespace: {
        mode: 'include',
        values: [dataStreamStat.namespace]
      }
    },
    breakdownField
  };
  const urlToLogsExplorer = locator.getRedirectUrl(params);
  const navigateToLogsExplorer = () => {
    locator.navigate(params);
  };
  const logsExplorerLinkProps = (0, _routerUtils.getRouterLinkProps)({
    href: urlToLogsExplorer,
    onClick: navigateToLogsExplorer
  });
  return {
    routerLinkProps: logsExplorerLinkProps,
    navigate: navigateToLogsExplorer
  };
};
const buildDiscoverConfig = ({
  locatorClient,
  dataStreamStat,
  query,
  from,
  to,
  breakdownField
}) => {
  const dataViewId = `${dataStreamStat.type}-${dataStreamStat.name}-*`;
  const dataViewTitle = dataStreamStat.integration ? `[${dataStreamStat.integration.title}] ${dataStreamStat.name}` : `${dataViewId}`;
  const params = {
    timeRange: {
      from,
      to
    },
    refreshInterval: {
      pause: true,
      value: 60000
    },
    dataViewId,
    dataViewSpec: {
      id: dataViewId,
      title: dataViewTitle
    },
    query,
    breakdownField,
    columns: ['@timestamp', 'message'],
    filters: [(0, _esQuery.buildPhraseFilter)({
      name: 'data_stream.namespace',
      type: 'string'
    }, dataStreamStat.namespace, {
      id: dataViewId,
      title: dataViewTitle
    })],
    interval: 'auto',
    sort: [['@timestamp', 'desc']]
  };
  const locator = locatorClient.get(_common.DISCOVER_APP_LOCATOR);
  const urlToDiscover = locator === null || locator === void 0 ? void 0 : locator.getRedirectUrl(params);
  const navigateToDiscover = () => {
    locator === null || locator === void 0 ? void 0 : locator.navigate(params);
  };
  const discoverLinkProps = (0, _routerUtils.getRouterLinkProps)({
    href: urlToDiscover,
    onClick: navigateToDiscover
  });
  return {
    routerLinkProps: discoverLinkProps,
    navigate: navigateToDiscover
  };
};