"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQLOverallStats = void 0;
var _common = require("@kbn/data-plugin/common");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _lodash = require("lodash");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _esqlUtils = require("@kbn/esql-utils");
var _i18n = require("@kbn/i18n");
var _esql_utils = require("../requests/esql_utils");
var _index_data_visualizer_viewer = require("../../constants/index_data_visualizer_viewer");
var _promise_all_settled_utils = require("../../../common/util/promise_all_settled_utils");
var _handle_error = require("./handle_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getESQLOverallStatsInChunk = async ({
  runRequest,
  fields,
  esqlBaseQueryWithLimit,
  filter,
  limitSize,
  totalCount,
  onError
}) => {
  if (fields.length > 0) {
    const aggToIndex = {
      count: 0,
      cardinality: 1
    };
    // Track what's the starting index for the next field
    // For aggregatable field, we are getting count(EVAL MV_MIN()) and count_disticnt
    // For non-aggregatable field, we are getting only count()
    let startIndex = 0;
    /** Example query:
     * from {indexPattern} | LIMIT {limitSize}
     * | STATs `{aggregableField}_count` = COUNT(MV_MIN(`{aggregableField}`)),
     * `{aggregableField}_cardinality` = COUNT_DISTINCT({aggregableField}),
     * `{nonAggregableField}_count` = COUNT({nonAggregableField})
     */
    const fieldsToFetch = fields.map(field => {
      if (field.aggregatable) {
        const result = {
          ...field,
          startIndex,
          // Field values can be an array of values (fieldName = ['a', 'b', 'c'])
          // and count(fieldName) will count all the field values in the array
          // Ex: for 2 docs, count(fieldName) might return 5
          // So we need to do count(EVAL(MV_MIN(fieldName))) instead
          // to get accurate % of rows where field value exists
          query: `${(0, _esql_utils.getSafeESQLName)(`${field.name}_count`)} = COUNT(MV_MIN(${(0, _esql_utils.getSafeESQLName)(field.name)})), ${(0, _esql_utils.getSafeESQLName)(`${field.name}_cardinality`)} = COUNT_DISTINCT(${(0, _esql_utils.getSafeESQLName)(field.name)})`
        };
        // +2 for count, and count_dictinct
        startIndex += 2;
        return result;
      } else {
        const result = {
          ...field,
          startIndex,
          query: `${(0, _esql_utils.getSafeESQLName)(`${field.name}_count`)} = COUNT(${(0, _esql_utils.getSafeESQLName)(field.name)})`
        };
        // +1 for count for non-aggregatable field
        startIndex += 1;
        return result;
      }
    });
    let countQuery = fieldsToFetch.length > 0 ? '| STATS ' : '';
    countQuery += fieldsToFetch.map(field => field.query).join(',');
    const query = (0, _esqlUtils.appendToESQLQuery)(esqlBaseQueryWithLimit, countQuery);
    const request = {
      params: {
        query,
        ...(filter ? {
          filter
        } : {})
      }
    };
    try {
      const esqlResults = await runRequest(request, {
        strategy: _common.ESQL_ASYNC_SEARCH_STRATEGY
      });
      const stats = {
        aggregatableExistsFields: [],
        aggregatableNotExistsFields: [],
        nonAggregatableExistsFields: [],
        nonAggregatableNotExistsFields: []
      };
      if (!esqlResults) {
        return;
      }
      const esqlResultsResp = esqlResults.rawResponse;
      const sampleCount = !(0, _mlIsDefined.isDefined)(limitSize) ? totalCount : limitSize;
      fieldsToFetch.forEach((field, idx) => {
        const count = esqlResultsResp.values[0][field.startIndex + aggToIndex.count];
        if (field.aggregatable === true) {
          const cardinality = esqlResultsResp.values[0][field.startIndex + aggToIndex.cardinality];
          if (count > 0) {
            stats.aggregatableExistsFields.push({
              ...field,
              fieldName: field.name,
              existsInDocs: true,
              stats: {
                sampleCount,
                count,
                cardinality
              }
            });
          } else {
            stats.aggregatableNotExistsFields.push({
              ...field,
              fieldName: field.name,
              existsInDocs: false,
              stats: undefined
            });
          }
        } else {
          if (count > 0) {
            stats.nonAggregatableExistsFields.push({
              fieldName: field.name,
              existsInDocs: true,
              stats: {
                sampleCount,
                count
              }
            });
          } else {
            stats.nonAggregatableNotExistsFields.push({
              fieldName: field.name,
              existsInDocs: false
            });
          }
        }
      });
      return stats;
    } catch (error) {
      (0, _handle_error.handleError)({
        error,
        request,
        onError,
        title: _i18n.i18n.translate('xpack.dataVisualizer.esql.countAndCardinalityError', {
          defaultMessage: 'Unable to fetch count & cardinality for {count} {count, plural, one {field} other {fields}}: {fieldNames}',
          values: {
            count: fieldsToFetch.length,
            fieldNames: fieldsToFetch.map(r => r.name).join()
          }
        })
      });
      return Promise.reject(error);
    }
  }
};

/**
 * Fetching count and cardinality in chunks of 30 fields per request in parallel
 * limiting at 10 requests maximum at a time
 * @param runRequest
 * @param fields
 * @param esqlBaseQueryWithLimit
 */
const getESQLOverallStats = async ({
  runRequest,
  fields,
  esqlBaseQueryWithLimit,
  filter,
  limitSize,
  totalCount,
  onError
}) => {
  const limiter = (0, _pLimit.default)(_index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS);
  const chunkedFields = (0, _lodash.chunk)(fields, 30);
  const resp = await Promise.allSettled(chunkedFields.map((groupedFields, idx) => limiter(() => getESQLOverallStatsInChunk({
    runRequest,
    fields: groupedFields,
    esqlBaseQueryWithLimit,
    limitSize,
    filter,
    totalCount,
    onError
  }))));
  const results = resp.filter(_promise_all_settled_utils.isFulfilled).map(r => r.value);
  const stats = results.reduce((acc, result) => {
    if (acc && result) {
      acc.aggregatableExistsFields.push(...result.aggregatableExistsFields);
      acc.aggregatableNotExistsFields.push(...result.aggregatableNotExistsFields);
      acc.nonAggregatableExistsFields.push(...result.nonAggregatableExistsFields);
      acc.nonAggregatableNotExistsFields.push(...result.nonAggregatableNotExistsFields);
    }
    return acc;
  }, {
    aggregatableExistsFields: [],
    aggregatableNotExistsFields: [],
    nonAggregatableExistsFields: [],
    nonAggregatableNotExistsFields: []
  });
  return stats;
};
exports.getESQLOverallStats = getESQLOverallStats;