"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useESQLOverallStatsData = exports.getInitialData = void 0;
var _common = require("@kbn/data-plugin/common");
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _mlCancellableSearch = require("@kbn/ml-cancellable-search");
var _esqlUtils = require("@kbn/esql-utils");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _constants = require("@kbn/unified-field-list/src/constants");
var _constants2 = require("../../../../../common/constants");
var _get_field_names = require("../../../common/components/fields_stats_grid/get_field_names");
var _kibana_context = require("../../../kibana_context");
var _progress_utils = require("../../progress_utils");
var _esql_utils = require("../../search_strategy/requests/esql_utils");
var _get_count_and_cardinality = require("../../search_strategy/esql_requests/get_count_and_cardinality");
var _handle_error = require("../../search_strategy/esql_requests/handle_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getESQLDocumentCountStats = async (runRequest, query, filter, timeFieldName, intervalMs, searchOptions, onError) => {
  if (!(0, _esql_utils.isESQLQuery)(query)) {
    throw Error(_i18n.i18n.translate('xpack.dataVisualizer.esql.noQueryProvided', {
      defaultMessage: 'No ES|QL query provided'
    }));
  }
  const esqlBaseQuery = query.esql;
  let earliestMs = Infinity;
  let latestMs = -Infinity;
  if (timeFieldName) {
    const aggQuery = (0, _esqlUtils.appendToESQLQuery)(esqlBaseQuery, ` | EVAL _timestamp_= TO_DOUBLE(DATE_TRUNC(${intervalMs} millisecond, ${(0, _esql_utils.getSafeESQLName)(timeFieldName)}))
    | stats rows = count(*) by _timestamp_`);
    const request = {
      params: {
        query: aggQuery,
        ...(filter ? {
          filter
        } : {})
      }
    };
    try {
      const esqlResults = await runRequest(request, {
        ...(searchOptions !== null && searchOptions !== void 0 ? searchOptions : {}),
        strategy: _common.ESQL_ASYNC_SEARCH_STRATEGY
      });
      let totalCount = 0;
      const _buckets = {};
      // @ts-expect-error ES types needs to be updated with columns and values as part of esql response
      esqlResults === null || esqlResults === void 0 ? void 0 : esqlResults.rawResponse.values.forEach(val => {
        const [count, bucket] = val;
        _buckets[bucket] = count;
        totalCount += count;
        if (bucket < earliestMs) {
          earliestMs = bucket;
        }
        if (bucket >= latestMs) {
          latestMs = bucket;
        }
      });
      const result = {
        interval: intervalMs,
        probability: 1,
        randomlySampled: false,
        timeRangeEarliest: earliestMs,
        timeRangeLatest: latestMs,
        buckets: _buckets,
        totalCount
      };
      return {
        documentCountStats: result,
        totalCount,
        request
      };
    } catch (error) {
      (0, _handle_error.handleError)({
        request,
        error,
        onError,
        title: _i18n.i18n.translate('xpack.dataVisualizer.esql.docCountError', {
          defaultMessage: `Error getting total count & doc count chart for ES|QL time-series data for request:`
        })
      });
      return Promise.reject(error);
    }
  } else {
    //  If not time field, get the total count
    const request = {
      params: {
        query: (0, _esqlUtils.appendToESQLQuery)(esqlBaseQuery, ' | STATS _count_ = COUNT(*)  | LIMIT 1'),
        ...(filter ? {
          filter
        } : {})
      }
    };
    try {
      const esqlResults = await runRequest(request, {
        ...(searchOptions !== null && searchOptions !== void 0 ? searchOptions : {}),
        strategy: _common.ESQL_ASYNC_SEARCH_STRATEGY
      });
      return {
        request,
        documentCountStats: undefined,
        totalCount: esqlResults === null || esqlResults === void 0 ? void 0 : esqlResults.rawResponse.values[0][0]
      };
    } catch (error) {
      (0, _handle_error.handleError)({
        request,
        error,
        title: _i18n.i18n.translate('xpack.dataVisualizer.esql.docCountNoneTimeseriesError', {
          defaultMessage: `Error getting total count for ES|QL data:`
        })
      });
      return Promise.reject(error);
    }
  }
};
const getInitialData = () => ({
  timeFieldName: undefined,
  columns: undefined,
  totalCount: undefined,
  exampleDocs: undefined,
  totalFields: undefined
});
exports.getInitialData = getInitialData;
const NON_AGGREGATABLE_FIELD_TYPES = new Set([_common.KBN_FIELD_TYPES.GEO_SHAPE, _common.KBN_FIELD_TYPES.GEO_POINT, _common.KBN_FIELD_TYPES.HISTOGRAM]);
const fieldStatsErrorTitle = _i18n.i18n.translate('xpack.dataVisualizer.index.errorFetchingESQLFieldStatisticsMessage', {
  defaultMessage: 'Error fetching field statistics for ES|QL query'
});
const useESQLOverallStatsData = fieldStatsRequest => {
  const {
    services: {
      data,
      notifications: {
        toasts
      }
    }
  } = (0, _kibana_context.useDataVisualizerKibana)();
  const previousExecutionTs = (0, _react.useRef)(undefined);
  const previousDocCountRequest = (0, _react.useRef)('');
  const previousError = (0, _react.useRef)();
  const {
    runRequest,
    cancelRequest
  } = (0, _mlCancellableSearch.useCancellableSearch)(data);
  const [tableData, setTableData] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), getInitialData());
  const [queryHistoryStatus, setQueryHistoryStatus] = (0, _react.useState)(false);
  const [overallStatsProgress, setOverallStatsProgress] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), (0, _progress_utils.getInitialProgress)());
  const onError = (0, _react.useCallback)((error, title) => {
    // If a previous error already occured, no need to show the toast again
    if (previousError.current) {
      return;
    }
    previousError.current = error;
    toasts.addError(error, {
      title: title !== null && title !== void 0 ? title : fieldStatsErrorTitle
    });
  }, [toasts]);
  const startFetch = (0, _react.useCallback)(async function fetchOverallStats() {
    try {
      var _columnsResp$rawRespo;
      if (!fieldStatsRequest || fieldStatsRequest.lastRefresh === 0) {
        return;
      }

      // Prevent requests from being called again when user clicks refresh consecutively too fast
      // or when Discover forces a refresh right after query or filter changes
      if (fieldStatsRequest.id === undefined && previousExecutionTs.current !== undefined && (fieldStatsRequest.lastRefresh === previousExecutionTs.current || fieldStatsRequest.lastRefresh - previousExecutionTs.current < 800)) {
        return;
      }
      previousExecutionTs.current = fieldStatsRequest.lastRefresh;
      cancelRequest();
      setOverallStatsProgress({
        ...(0, _progress_utils.getInitialProgress)(),
        isRunning: true,
        error: undefined
      });
      previousError.current = undefined;
      const {
        searchQuery,
        intervalMs,
        filter: filter,
        limit,
        totalCount: knownTotalCount
      } = fieldStatsRequest;
      if (!(0, _esql_utils.isESQLQuery)(searchQuery)) {
        return;
      }
      const intervalInMs = intervalMs === 0 ? 60 * 60 * 60 * 10 : intervalMs;

      // For doc count chart, we want the full base query without any limit
      const esqlBaseQuery = searchQuery.esql;
      setQueryHistoryStatus(true);

      // Note: dropNullColumns will return empty [] for all_columns if limit size is 0
      // So we are making a query with default limit
      // And use this one query to
      // 1) identify populated/empty fields
      // 2) gather examples for populated text fields
      const columnsResp = await runRequest({
        params: {
          // Doing this to match with the default limit
          query: (0, _esqlUtils.getESQLWithSafeLimit)(esqlBaseQuery, _constants.ESQL_SAFE_LIMIT),
          ...(filter ? {
            filter
          } : {}),
          dropNullColumns: true
        }
      }, {
        strategy: _common.ESQL_ASYNC_SEARCH_STRATEGY
      });
      setQueryHistoryStatus(false);
      const columnInfo = columnsResp !== null && columnsResp !== void 0 && columnsResp.rawResponse ? (_columnsResp$rawRespo = columnsResp.rawResponse.all_columns) !== null && _columnsResp$rawRespo !== void 0 ? _columnsResp$rawRespo : columnsResp.rawResponse.columns : [];
      const populatedColumns = new Set(columnsResp === null || columnsResp === void 0 ? void 0 : columnsResp.rawResponse.columns.map(c => c.name));
      const columns = columnInfo.map(c => ({
        ...c,
        secondaryType: (0, _get_field_names.getSupportedFieldType)(c.type)
      }));
      const timeFields = columns.filter(d => d.type === 'date');
      const dataViewTimeField = timeFields.find(f => f.name === (fieldStatsRequest === null || fieldStatsRequest === void 0 ? void 0 : fieldStatsRequest.timeFieldName)) ? fieldStatsRequest === null || fieldStatsRequest === void 0 ? void 0 : fieldStatsRequest.timeFieldName : undefined;

      // If a date field named '@timestamp' exists, set that as default time field
      // Else, use the default time view defined by data view
      // Else, use first available date field as default
      const timeFieldName = timeFields.length > 0 ? timeFields.find(f => f.name === '@timestamp') ? '@timestamp' : dataViewTimeField !== null && dataViewTimeField !== void 0 ? dataViewTimeField : timeFields[0].name : undefined;
      setTableData({
        columns,
        timeFieldName
      });

      // We don't need to fetch the doc count stats again if only the limit size is changed
      // so return the previous totalCount, documentCountStats if available
      const hashedDocCountParams = JSON.stringify({
        searchQuery,
        filter,
        timeFieldName,
        intervalInMs
      });
      let {
        totalCount,
        documentCountStats
      } = tableData;
      if (knownTotalCount !== undefined) {
        totalCount = knownTotalCount;
      }
      if (knownTotalCount === undefined && (totalCount === undefined || documentCountStats === undefined || hashedDocCountParams !== previousDocCountRequest.current)) {
        setTableData({
          totalCount: undefined,
          documentCountStats: undefined
        });
        previousDocCountRequest.current = hashedDocCountParams;
        const results = await getESQLDocumentCountStats(runRequest, searchQuery, filter, timeFieldName, intervalInMs, undefined, onError);
        totalCount = results.totalCount;
        documentCountStats = results.documentCountStats;
        setTableData({
          totalCount,
          documentCountStats
        });
      }
      if (totalCount === undefined) {
        totalCount = 0;
      }
      setOverallStatsProgress({
        loaded: 50
      });
      const aggregatableNotExistsFields = [];
      const nonAggregatableNotExistsFields = [];
      const fields = columns
      // Some field types are not supported by ESQL yet
      // Also, temporarily removing null columns because it causes problems with some aggs
      // See https://github.com/elastic/elasticsearch/issues/104430
      .filter(c => c.type !== 'unsupported' && c.type !== 'null').map(field => {
        return {
          ...field,
          aggregatable: !NON_AGGREGATABLE_FIELD_TYPES.has(field.type)
        };
      });
      const populatedFields = fields.filter(field => populatedColumns.has(field.name));
      fields === null || fields === void 0 ? void 0 : fields.forEach(field => {
        const fieldName = field.name;
        if (!_constants2.OMIT_FIELDS.includes(fieldName)) {
          if (!field.aggregatable) {
            if (!populatedColumns.has(fieldName)) {
              nonAggregatableNotExistsFields.push({
                ...field,
                fieldName: field.name,
                secondaryType: (0, _get_field_names.getSupportedFieldType)(field.type),
                existsInDocs: false
              });
            }
          } else {
            if (!populatedColumns.has(fieldName)) {
              aggregatableNotExistsFields.push({
                ...field,
                fieldName: field.name,
                aggregatable: true,
                existsInDocs: false
              });
            }
          }
        }
      });

      // COUNT + CARDINALITY
      // For % count & cardinality, we want the full base query WITH specified limit
      // to safeguard against huge datasets
      const esqlBaseQueryWithLimit = (0, _esqlUtils.getESQLWithSafeLimit)(searchQuery.esql, limit);
      if (totalCount === 0) {
        setTableData({
          aggregatableFields: undefined,
          nonAggregatableFields: undefined,
          overallStats: undefined,
          columns: undefined,
          timeFieldName: undefined
        });
        setOverallStatsProgress({
          loaded: 100,
          isRunning: false,
          error: undefined
        });
        return;
      }
      if (totalCount > 0 && fields.length > 0) {
        const stats = await (0, _get_count_and_cardinality.getESQLOverallStats)({
          runRequest,
          // Only need to fetch stats for fields we know are populated
          fields: populatedFields,
          esqlBaseQueryWithLimit,
          filter,
          limitSize: limit,
          totalCount,
          onError
        });
        if (!stats) return;
        stats.aggregatableNotExistsFields = aggregatableNotExistsFields;
        stats.nonAggregatableNotExistsFields = nonAggregatableNotExistsFields;
        setTableData({
          overallStats: stats,
          totalFields: columns.length
        });
        setOverallStatsProgress({
          loaded: 100,
          isRunning: false,
          error: undefined
        });
        const columnsWithExamples = columnInfo.reduce((hashmap, curr, idx) => {
          if (curr.type === 'text' || curr.type === 'geo_point' || curr.type === 'geo_shape') {
            hashmap[curr.name] = idx;
          }
          return hashmap;
        }, {});
        const exampleDocs = Object.entries(columnsWithExamples).map(([fieldName, idx]) => {
          var _columnsResp$rawRespo2;
          const examples = [...new Set(columnsResp === null || columnsResp === void 0 ? void 0 : (_columnsResp$rawRespo2 = columnsResp.rawResponse) === null || _columnsResp$rawRespo2 === void 0 ? void 0 : _columnsResp$rawRespo2.values.map(row => row[idx]))].filter(_mlIsDefined.isDefined).slice(0, 10);
          return {
            fieldName,
            examples: examples
          };
        });
        setTableData({
          exampleDocs
        });
      }
    } catch (error) {
      setOverallStatsProgress({
        loaded: 100,
        isRunning: false,
        error
      });
      setQueryHistoryStatus(false);
      // If error already handled in sub functions, no need to propogate
      if (error.name !== 'AbortError' && error.handled !== true) {
        toasts.addError(error, {
          title: fieldStatsErrorTitle
        });
      }
      // Log error to console for better debugging
      // eslint-disable-next-line no-console
      console.error(`${fieldStatsErrorTitle}: fetchOverallStats`, error);
    }
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [JSON.stringify({
    fieldStatsRequest
  })]);

  // auto-update
  (0, _react.useEffect)(() => {
    startFetch();
  }, [startFetch]);
  return (0, _react.useMemo)(() => ({
    ...tableData,
    overallStatsProgress,
    cancelOverallStatsRequest: cancelRequest,
    queryHistoryStatus
  }), [tableData, overallStatsProgress, cancelRequest, queryHistoryStatus]);
};
exports.useESQLOverallStatsData = useESQLOverallStatsData;