"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateHash = calculateHash;
exports.registerFields = void 0;
var _crypto = require("crypto");
var _utils = require("../../../common/utils");
var _fetcher = require("../../fetcher");
var _constants = require("../../../common/constants");
var _fields_for = require("./fields_for");
var _constants2 = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function calculateHash(srcBuffer) {
  const hash = (0, _crypto.createHash)('sha1');
  hash.update(srcBuffer);
  return hash.digest('hex');
}
const handler = isRollupsEnabled => async (context, request, response) => {
  const core = await context.core;
  const uiSettings = core.uiSettings.client;
  const {
    asCurrentUser
  } = core.elasticsearch.client;
  const indexPatterns = new _fetcher.IndexPatternsFetcher(asCurrentUser, {
    uiSettingsClient: uiSettings,
    rollupsEnabled: isRollupsEnabled()
  });
  const {
    pattern,
    meta_fields: metaFields,
    type,
    rollup_index: rollupIndex,
    allow_no_index: allowNoIndex,
    include_unmapped: includeUnmapped,
    field_types: fieldTypes
  } = request.query;
  let parsedFields = [];
  let parsedMetaFields = [];
  let parsedFieldTypes = [];
  try {
    var _request$query$fields;
    parsedMetaFields = (0, _fields_for.parseFields)(metaFields, 'meta_fields');
    parsedFields = (0, _fields_for.parseFields)((_request$query$fields = request.query.fields) !== null && _request$query$fields !== void 0 ? _request$query$fields : [], 'fields');
    parsedFieldTypes = (0, _fields_for.parseFields)(fieldTypes || [], 'field_types');
  } catch (error) {
    return response.badRequest();
  }
  try {
    const {
      fields,
      indices
    } = await indexPatterns.getFieldsForWildcard({
      pattern,
      metaFields: parsedMetaFields,
      type,
      rollupIndex,
      fieldCapsOptions: {
        allow_no_indices: allowNoIndex || false,
        includeUnmapped
      },
      fieldTypes: parsedFieldTypes,
      ...(parsedFields.length > 0 ? {
        fields: parsedFields
      } : {})
    });
    const body = {
      fields,
      indices
    };
    const bodyAsString = JSON.stringify(body);
    const etag = calculateHash(Buffer.from(bodyAsString));
    const headers = {
      'content-type': 'application/json',
      etag,
      vary: 'accept-encoding, user-hash'
    };

    // field cache is configurable in classic environment but not on serverless
    let cacheMaxAge = _constants2.DEFAULT_FIELD_CACHE_FRESHNESS;
    const cacheMaxAgeSetting = await uiSettings.get('data_views:cache_max_age');
    if (cacheMaxAgeSetting !== undefined) {
      cacheMaxAge = cacheMaxAgeSetting;
    }
    if (cacheMaxAge && fields.length) {
      const stale = 365 * 24 * 60 * 60 - cacheMaxAge;
      headers['cache-control'] = `private, max-age=${cacheMaxAge}, stale-while-revalidate=${stale}`;
    } else {
      headers['cache-control'] = 'private, no-cache';
    }
    const ifNoneMatch = request.headers['if-none-match'];
    const ifNoneMatchString = Array.isArray(ifNoneMatch) ? ifNoneMatch[0] : ifNoneMatch;
    if (ifNoneMatchString) {
      const requestHash = (0, _utils.unwrapEtag)(ifNoneMatchString);
      if (etag === requestHash) {
        return response.notModified({
          headers
        });
      }
    }
    return response.ok({
      body: bodyAsString,
      headers
    });
  } catch (error) {
    var _error$output, _error$output2;
    if (typeof error === 'object' && !!(error !== null && error !== void 0 && error.isBoom) && !!(error !== null && error !== void 0 && (_error$output = error.output) !== null && _error$output !== void 0 && _error$output.payload) && typeof (error === null || error === void 0 ? void 0 : (_error$output2 = error.output) === null || _error$output2 === void 0 ? void 0 : _error$output2.payload) === 'object') {
      var _error$output3;
      const payload = error === null || error === void 0 ? void 0 : (_error$output3 = error.output) === null || _error$output3 === void 0 ? void 0 : _error$output3.payload;
      return response.notFound({
        body: {
          message: payload.message,
          attributes: payload
        }
      });
    } else {
      return response.notFound();
    }
  }
};
const registerFields = (router, getStartServices, isRollupsEnabled) => {
  router.versioned.get({
    path: _constants.FIELDS_PATH,
    access: 'internal',
    enableQueryVersion: true
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _fields_for.querySchema
      },
      response: _fields_for.validate.response
    }
  }, handler(isRollupsEnabled));
};
exports.registerFields = registerFields;