"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataViewLazy = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fieldTypes = require("@kbn/field-types");
var _lodash = require("lodash");
var _common = require("@kbn/kibana-utils-plugin/common");
var _abstract_data_views = require("./abstract_data_views");
var _fields = require("../fields");
var _data_view_lazy_util = require("./data_view_lazy_util");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class DataViewLazy extends _abstract_data_views.AbstractDataView {
  constructor(config) {
    super(config);
    (0, _defineProperty2.default)(this, "apiClient", void 0);
    (0, _defineProperty2.default)(this, "fieldCache", new Map());
    /**
     * Returns true if scripted fields are enabled
     */
    (0, _defineProperty2.default)(this, "scriptedFieldsEnabled", false);
    (0, _defineProperty2.default)(this, "getRuntimeFields", ({
      fieldName = ['*']
    }) =>
    // getRuntimeFieldSpecMap flattens composites into a list of fields
    Object.values(this.getRuntimeFieldSpecMap({
      fieldName
    })).reduce((col, field) => {
      var _cachedField;
      if (!(0, _data_view_lazy_util.fieldMatchesFieldsRequested)(field.name, fieldName)) {
        return col;
      }
      let cachedField = this.fieldCache.get(field.name);
      // if mapped field, can't be runtime field
      if ((_cachedField = cachedField) !== null && _cachedField !== void 0 && _cachedField.isMapped) {
        return col;
      }
      if (!cachedField) {
        var _this$fieldAttrs, _this$fieldAttrs$fiel, _this$fieldAttrs2, _this$fieldAttrs2$fie, _this$fieldAttrs3, _this$fieldAttrs3$fie;
        cachedField = new _fields.DataViewField({
          ...field,
          count: (_this$fieldAttrs = this.fieldAttrs) === null || _this$fieldAttrs === void 0 ? void 0 : (_this$fieldAttrs$fiel = _this$fieldAttrs[field.name]) === null || _this$fieldAttrs$fiel === void 0 ? void 0 : _this$fieldAttrs$fiel.count,
          customLabel: (_this$fieldAttrs2 = this.fieldAttrs) === null || _this$fieldAttrs2 === void 0 ? void 0 : (_this$fieldAttrs2$fie = _this$fieldAttrs2[field.name]) === null || _this$fieldAttrs2$fie === void 0 ? void 0 : _this$fieldAttrs2$fie.customLabel,
          customDescription: (_this$fieldAttrs3 = this.fieldAttrs) === null || _this$fieldAttrs3 === void 0 ? void 0 : (_this$fieldAttrs3$fie = _this$fieldAttrs3[field.name]) === null || _this$fieldAttrs3$fie === void 0 ? void 0 : _this$fieldAttrs3$fie.customDescription,
          shortDotsEnable: this.shortDotsEnable
        });
        this.fieldCache.set(field.name, cachedField);
      }
      col[field.name] = cachedField;
      return col;
    }, {}));
    (0, _defineProperty2.default)(this, "getRuntimeFieldSpecMap", ({
      fieldName = ['*']
    }) => {
      const spec = {};
      const addRuntimeFieldToSpecFields = (name, fieldType, runtimeField, parentName) => {
        var _this$fieldAttrs4, _this$fieldAttrs4$nam, _this$fieldAttrs5, _this$fieldAttrs5$nam, _this$fieldAttrs6, _this$fieldAttrs6$nam;
        spec[name] = {
          name,
          type: (0, _fieldTypes.castEsToKbnFieldTypeName)(fieldType),
          esTypes: [fieldType],
          runtimeField,
          aggregatable: true,
          searchable: true,
          readFromDocValues: false,
          customLabel: (_this$fieldAttrs4 = this.fieldAttrs) === null || _this$fieldAttrs4 === void 0 ? void 0 : (_this$fieldAttrs4$nam = _this$fieldAttrs4[name]) === null || _this$fieldAttrs4$nam === void 0 ? void 0 : _this$fieldAttrs4$nam.customLabel,
          customDescription: (_this$fieldAttrs5 = this.fieldAttrs) === null || _this$fieldAttrs5 === void 0 ? void 0 : (_this$fieldAttrs5$nam = _this$fieldAttrs5[name]) === null || _this$fieldAttrs5$nam === void 0 ? void 0 : _this$fieldAttrs5$nam.customDescription,
          count: (_this$fieldAttrs6 = this.fieldAttrs) === null || _this$fieldAttrs6 === void 0 ? void 0 : (_this$fieldAttrs6$nam = _this$fieldAttrs6[name]) === null || _this$fieldAttrs6$nam === void 0 ? void 0 : _this$fieldAttrs6$nam.count
        };
        if (parentName) {
          spec[name].parentName = parentName;
        }
      };

      // CREATE RUNTIME FIELDS
      for (const [name, runtimeField] of Object.entries(this.runtimeFieldMap || {})) {
        if ((0, _data_view_lazy_util.fieldMatchesFieldsRequested)(name, fieldName)) {
          // For composite runtime field we add the subFields, **not** the composite
          if (runtimeField.type === 'composite') {
            Object.entries(runtimeField.fields).forEach(([subFieldName, subField]) => {
              addRuntimeFieldToSpecFields(`${name}.${subFieldName}`, subField.type, runtimeField, name);
            });
          } else {
            addRuntimeFieldToSpecFields(name, runtimeField.type, runtimeField);
          }
        }
      }
      return spec;
    });
    (0, _defineProperty2.default)(this, "isTimeBased", async () => !!(await this.getTimeField()));
    (0, _defineProperty2.default)(this, "getTimeField", () => this.timeFieldName ? this.getFieldByName(this.timeFieldName) : undefined);
    this.apiClient = config.apiClient;
    this.scriptedFieldsEnabled = config.scriptedFieldsEnabled;
  }
  async getFields({
    mapped = true,
    scripted = true,
    runtime = true,
    fieldName,
    forceRefresh = false,
    unmapped,
    indexFilter,
    metaFields = true
  }) {
    let mappedResult = {};
    let scriptedResult = {};
    let runtimeResult = {};
    // need to know if runtime fields are also mapped
    if (mapped || runtime) {
      // if we just need runtime fields, we can ask for the set of runtime fields specifically
      const fieldsToRequest = mapped ? fieldName : (0, _data_view_lazy_util.fieldsMatchFieldsRequested)(Object.keys(this.runtimeFieldMap), fieldName);
      mappedResult = await this.getMappedFields({
        fieldName: fieldsToRequest,
        forceRefresh,
        unmapped,
        indexFilter,
        metaFields
      });
    }
    if (scripted && this.scriptedFieldsEnabled) {
      scriptedResult = this.getScriptedFields({
        fieldName
      });
    }
    if (runtime) {
      runtimeResult = this.getRuntimeFields({
        fieldName
      });
    }
    const fieldMap = {
      ...mappedResult,
      ...scriptedResult,
      ...runtimeResult
    };
    let fieldMapSorted = {};
    let hasBeenSorted = false;
    return {
      getFieldMap: () => fieldMap,
      getFieldMapSorted: () => {
        if (!hasBeenSorted) {
          fieldMapSorted = (0, _lodash.chain)(fieldMap).toPairs().sortBy(0).fromPairs().value();
          hasBeenSorted = true;
        }
        return fieldMapSorted;
      }
    };
  }
  /**
   * Add a runtime field - Appended to existing mapped field or a new field is
   * created as appropriate.
   * @param name Field name
   * @param runtimeField Runtime field definition
   */
  async addRuntimeField(name, runtimeField) {
    if (name.includes('*')) {
      throw new _common.CharacterNotAllowedInField('*', name);
    }
    const {
      type,
      script,
      customLabel,
      customDescription,
      format,
      popularity
    } = runtimeField;
    if (type === 'composite') {
      return this.addCompositeRuntimeField(name, runtimeField);
    }
    this.addRuntimeFieldInteral(name, runtimeField);
    const field = await this.updateOrAddRuntimeField(name, type, {
      type,
      script
    }, {
      customLabel,
      customDescription,
      format,
      popularity
    });
    return [field];
  }
  async addCompositeRuntimeField(name, runtimeField) {
    const {
      fields
    } = runtimeField;

    // Make sure subFields are provided
    if (fields === undefined || Object.keys(fields).length === 0) {
      throw new Error(`Can't add composite runtime field [name = ${name}] without subfields.`);
    }

    // Make sure no field with the same name already exist
    if ((await this.getFieldByName(name)) !== undefined) {
      throw new Error(`Can't create composite runtime field ["${name}"] as there is already a field with this name`);
    }

    // We first remove the runtime composite field with the same name which will remove all of its subFields.
    // This guarantees that we don't leave behind orphan data view fields
    this.removeRuntimeField(name);
    const runtimeFieldSpec = (0, _utils.removeFieldAttrs)(runtimeField);

    // We don't add composite runtime fields to the field list as
    // they are not fields but **holder** of fields.
    // What we do add to the field list are all their subFields.
    const dataViewFields = await Promise.all(Object.entries(fields).map(async ([subFieldName, subField]) =>
    // Every child field gets the complete runtime field script for consumption by searchSource
    await this.updateOrAddRuntimeField(`${name}.${subFieldName}`, subField.type, runtimeFieldSpec, {
      customLabel: subField.customLabel,
      customDescription: subField.customDescription,
      format: subField.format,
      popularity: subField.popularity
    })));
    this.addRuntimeFieldInteral(name, runtimeField);
    return dataViewFields;
  }

  /**
   * Remove a runtime field - removed from mapped field or removed unmapped
   * field as appropriate. Doesn't clear associated field attributes.
   * @param name - Field name to remove
   */
  removeRuntimeField(name) {
    const existingField = this.fieldCache.get(name);
    if (existingField && existingField.isMapped) {
      // mapped field, remove runtimeField definition
      existingField.runtimeField = undefined;
    } else {
      Object.values(this.getFieldsByRuntimeFieldName(name) || {}).forEach(field => {
        this.fieldCache.delete(field.name);
      });
    }
    this.removeRuntimeFieldInteral(name);
  }
  getFieldsByRuntimeFieldName(name) {
    const runtimeField = this.getRuntimeField(name);
    if (!runtimeField) {
      return;
    }
    if (runtimeField.type === 'composite') {
      return Object.entries(runtimeField.fields).reduce((acc, [subFieldName, subField]) => {
        const fieldFullName = `${name}.${subFieldName}`;
        const dataViewField = this.fieldCache.get(fieldFullName);
        if (!dataViewField) {
          // We should never enter here as all composite runtime subfield
          // are converted to data view fields.
          return acc;
        }
        acc[subFieldName] = dataViewField;
        return acc;
      }, {});
    }
    const primitveRuntimeField = this.fieldCache.get(name);
    return primitveRuntimeField && {
      [name]: primitveRuntimeField
    };
  }

  // creates / updates non composite runtime fields. only called from addRuntimeField or addCompositeRuntimeField
  async updateOrAddRuntimeField(fieldName, fieldType, runtimeFieldSpec, config) {
    if (fieldType === 'composite') {
      throw new Error(`Trying to add composite field as primitive field, this shouldn't happen! [name = ${fieldName}]`);
    }

    // Create the field if it does not exist or update an existing one
    let fld = await this.getFieldByName(fieldName);
    if (fld) {
      fld.runtimeField = runtimeFieldSpec;
      fld.spec.type = (0, _fieldTypes.castEsToKbnFieldTypeName)(fieldType);
      fld.spec.esTypes = [fieldType];
    } else {
      var _config$popularity;
      fld = new _fields.DataViewField({
        name: fieldName,
        runtimeField: runtimeFieldSpec,
        type: (0, _fieldTypes.castEsToKbnFieldTypeName)(fieldType),
        esTypes: [fieldType],
        aggregatable: true,
        searchable: true,
        count: (_config$popularity = config.popularity) !== null && _config$popularity !== void 0 ? _config$popularity : 0,
        readFromDocValues: false,
        shortDotsEnable: this.shortDotsEnable
      });
      this.fieldCache.set(fieldName, fld);
    }
    this.setFieldCustomLabel(fieldName, config.customLabel);
    this.setFieldCustomDescription(fieldName, config.customDescription);
    if (config.popularity || config.popularity === null) {
      this.setFieldCount(fieldName, config.popularity);
    }
    if (config.format) {
      this.setFieldFormat(fieldName, config.format);
    } else if (config.format === null) {
      this.deleteFieldFormat(fieldName);
    }
    return fld;
  }
  getScriptedFields({
    fieldName = ['*']
  }) {
    const dataViewFields = {};
    Object.values(this.scriptedFieldsMap).forEach(field => {
      var _this$fieldAttrs7, _this$fieldAttrs7$fie, _this$fieldAttrs8, _this$fieldAttrs8$fie, _this$fieldAttrs9, _this$fieldAttrs9$fie;
      if (!(0, _data_view_lazy_util.fieldMatchesFieldsRequested)(field.name, fieldName)) {
        return;
      }
      let fld = this.fieldCache.get(field.name);

      // scripted field overrides mapped field
      if (fld && !fld.scripted && fld.isMapped) {
        this.fieldCache.delete(field.name);
      }
      fld = new _fields.DataViewField({
        ...field,
        scripted: true,
        searchable: true,
        aggregatable: true,
        count: (_this$fieldAttrs7 = this.fieldAttrs) === null || _this$fieldAttrs7 === void 0 ? void 0 : (_this$fieldAttrs7$fie = _this$fieldAttrs7[field.name]) === null || _this$fieldAttrs7$fie === void 0 ? void 0 : _this$fieldAttrs7$fie.count,
        customLabel: (_this$fieldAttrs8 = this.fieldAttrs) === null || _this$fieldAttrs8 === void 0 ? void 0 : (_this$fieldAttrs8$fie = _this$fieldAttrs8[field.name]) === null || _this$fieldAttrs8$fie === void 0 ? void 0 : _this$fieldAttrs8$fie.customLabel,
        customDescription: (_this$fieldAttrs9 = this.fieldAttrs) === null || _this$fieldAttrs9 === void 0 ? void 0 : (_this$fieldAttrs9$fie = _this$fieldAttrs9[field.name]) === null || _this$fieldAttrs9$fie === void 0 ? void 0 : _this$fieldAttrs9$fie.customDescription
      });
      this.fieldCache.set(field.name, fld);
      dataViewFields[field.name] = fld;
    });
    return dataViewFields;
  }
  async getMappedFields({
    fieldName,
    forceRefresh = false,
    unmapped: includeUnmapped,
    indexFilter,
    metaFields,
    fieldTypes
  }) {
    var _this$typeMeta, _this$typeMeta$params;
    const response = await this.apiClient.getFieldsForWildcard({
      pattern: this.getIndexPattern(),
      metaFields: metaFields ? this.metaFields : undefined,
      type: this.type,
      rollupIndex: (_this$typeMeta = this.typeMeta) === null || _this$typeMeta === void 0 ? void 0 : (_this$typeMeta$params = _this$typeMeta.params) === null || _this$typeMeta$params === void 0 ? void 0 : _this$typeMeta$params.rollup_index,
      fields: fieldName || ['*'],
      allowNoIndex: true,
      indexFilter,
      includeUnmapped,
      forceRefresh,
      fieldTypes
    });
    const dataViewFields = {};
    response.fields.forEach(field => {
      // keep existing field object, make sure content is fresh
      let fld = this.fieldCache.get(field.name);
      if (fld) {
        // get fresh attributes
        (0, _lodash.assign)(fld.spec, field);
        fld.spec.runtimeField = undefined; // unset if it was a runtime field but now mapped
        fld.spec.isMapped = true;
      } else {
        var _this$fieldAttrs10, _this$fieldAttrs10$fi, _this$fieldAttrs11, _this$fieldAttrs11$fi, _this$fieldAttrs12, _this$fieldAttrs12$fi;
        fld = new _fields.DataViewField({
          ...field,
          count: (_this$fieldAttrs10 = this.fieldAttrs) === null || _this$fieldAttrs10 === void 0 ? void 0 : (_this$fieldAttrs10$fi = _this$fieldAttrs10[field.name]) === null || _this$fieldAttrs10$fi === void 0 ? void 0 : _this$fieldAttrs10$fi.count,
          customLabel: (_this$fieldAttrs11 = this.fieldAttrs) === null || _this$fieldAttrs11 === void 0 ? void 0 : (_this$fieldAttrs11$fi = _this$fieldAttrs11[field.name]) === null || _this$fieldAttrs11$fi === void 0 ? void 0 : _this$fieldAttrs11$fi.customLabel,
          customDescription: (_this$fieldAttrs12 = this.fieldAttrs) === null || _this$fieldAttrs12 === void 0 ? void 0 : (_this$fieldAttrs12$fi = _this$fieldAttrs12[field.name]) === null || _this$fieldAttrs12$fi === void 0 ? void 0 : _this$fieldAttrs12$fi.customDescription,
          shortDotsEnable: this.shortDotsEnable
        });
        this.fieldCache.set(field.name, fld);
      }
      dataViewFields[field.name] = fld;
    });
    return dataViewFields;
  }
  getScriptedFieldsForQuery() {
    return Object.values(this.scriptedFieldsMap).reduce((scriptFields, field) => {
      scriptFields[field.name] = {
        script: {
          source: field.script,
          lang: field.lang
        }
      };
      return scriptFields;
    }, {});
  }
  async getComputedFields({
    fieldName = ['*']
  }) {
    const scriptFields = {};
    const fieldMap = (await this.getFields({
      fieldName,
      fieldTypes: ['date', 'date_nanos'],
      scripted: false,
      runtime: false
    })).getFieldMap();
    const docvalueFields = Object.values(fieldMap).map(dateField => {
      return {
        field: dateField.name,
        format: dateField.esTypes && dateField.esTypes.indexOf('date_nanos') !== -1 ? 'strict_date_time' : 'date_time'
      };
    });
    (0, _lodash.each)(this.getScriptedFields({
      fieldName
    }), function (field) {
      scriptFields[field.name] = {
        script: {
          source: field.script,
          lang: field.lang
        }
      };
    });
    return {
      scriptFields,
      docvalueFields,
      runtimeFields: this.getRuntimeMappings()
    };
  }
  getRuntimeMappings() {
    return this.runtimeFieldMap;
  }

  /**
   * Creates static representation of the data view.
   * @param includeFields Whether or not to include the `fields` list as part of this spec. If not included, the list
   * will be fetched from Elasticsearch when instantiating a new Data View with this spec.
   */
  async toSpec(params) {
    const spec = this.toSpecShared(!!(params !== null && params !== void 0 && params.fieldParams));
    if (params !== null && params !== void 0 && params.fieldParams) {
      const fields = {};
      const fieldMap = (await this.getFields(params.fieldParams)).getFieldMap();
      Object.values(fieldMap).forEach(field => {
        fields[field.name] = field.toSpec({
          getFormatterForField: this.getFormatterForField.bind(this)
        });
      });
      spec.fields = fields;
    }
    return spec;
  }

  /**
   * Creates a minimal static representation of the data view. Fields and popularity scores will be omitted.
   */
  // todo make shared
  toMinimalSpec(params) {
    var _params$keepFieldAttr;
    const fieldAttrsToKeep = (_params$keepFieldAttr = params === null || params === void 0 ? void 0 : params.keepFieldAttrs) !== null && _params$keepFieldAttr !== void 0 ? _params$keepFieldAttr : ['customLabel', 'customDescription'];
    // removes `fields`
    const spec = this.toSpecShared(false);
    if (spec.fieldAttrs) {
      // removes `count` props (popularity scores) from `fieldAttrs`
      spec.fieldAttrs = (0, _lodash.pickBy)(
      // removes unnecessary attributes
      (0, _lodash.mapValues)(spec.fieldAttrs, fieldAttrs => (0, _lodash.pick)(fieldAttrs, fieldAttrsToKeep)),
      // removes empty objects if all attributes have been removed
      trimmedFieldAttrs => Object.keys(trimmedFieldAttrs).length > 0);

      // removes `fieldAttrs` if it's empty
      if (Object.keys(spec.fieldAttrs).length === 0) {
        delete spec.fieldAttrs;
      }
    }
    return Object.fromEntries(Object.entries(spec).filter(([, v]) => typeof v !== 'undefined'));
  }

  /**
   * returns true if dataview contains TSDB fields
   */
  async isTSDBMode() {
    const fieldMap = (await this.getFields({
      fieldName: ['*']
    })).getFieldMap();
    return Object.values(fieldMap).some(field => field.timeSeriesDimension || field.timeSeriesMetric);
  }
  removeScriptedField(fieldName) {
    this.deleteScriptedFieldInternal(fieldName);
    this.fieldCache.delete(fieldName);
  }
  upsertScriptedField(field) {
    return this.upsertScriptedFieldInternal(field);
  }
  async isTimeNanosBased() {
    const field = await this.getTimeField();
    return field ? field.type === 'date_nanos' : false;
  }
  async getFieldByName(name, forceRefresh = false) {
    const fieldMap = (await this.getFields({
      fieldName: [name],
      forceRefresh
    })).getFieldMap();
    return fieldMap[name];
  }

  /**
   * Set field custom label
   * @param fieldName name of field to set custom label on
   * @param customLabel custom label value. If undefined, custom label is removed
   */

  setFieldCustomLabel(fieldName, customLabel) {
    const newCustomLabel = customLabel === null ? undefined : customLabel;
    this.setFieldCustomLabelInternal(fieldName, customLabel);
    const fieldObject = this.fieldCache.get(fieldName);
    if (fieldObject) {
      fieldObject.spec.customLabel = newCustomLabel;
    }
  }
  setFieldCount(fieldName, count) {
    const newCount = count === null ? undefined : count;
    this.setFieldCountInternal(fieldName, count);
    const fieldObject = this.fieldCache.get(fieldName);
    if (fieldObject) {
      if (!newCount) fieldObject.deleteCount();else fieldObject.spec.count = newCount;
    }
  }
  setFieldCustomDescription(fieldName, customDescription) {
    const newCustomDescription = customDescription === null ? undefined : customDescription;
    this.setFieldCustomDescriptionInternal(fieldName, customDescription);
    const fieldObject = this.fieldCache.get(fieldName);
    if (fieldObject) {
      fieldObject.spec.customDescription = newCustomDescription;
    }
  }
}
exports.DataViewLazy = DataViewLazy;