"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToEuiFilterOptions = convertToEuiFilterOptions;
exports.getPath = getPath;
exports.getTabs = getTabs;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _constants = require("../constants");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function filterByName(items, filter) {
  const lowercaseFilter = (filter || '').toLowerCase();
  return items.filter(item => item.name.toLowerCase().includes(lowercaseFilter));
}
function getCounts(fields, sourceFilters, fieldFilter = '') {
  const fieldCount = (0, _lodash.countBy)(filterByName(fields, fieldFilter), function (field) {
    return field.scripted ? 'scripted' : 'indexed';
  });
  (0, _lodash.defaults)(fieldCount, {
    indexed: 0,
    scripted: 0,
    sourceFilters: sourceFilters.excludes ? sourceFilters.excludes.filter(value => value.toLowerCase().includes(fieldFilter.toLowerCase())).length : 0
  });
  return fieldCount;
}
function getTitle(type, filteredCount, totalCount) {
  let title = '';
  switch (type) {
    case 'indexed':
      title = _i18n.i18n.translate('indexPatternManagement.editIndexPattern.tabs.fieldsHeader', {
        defaultMessage: 'Fields'
      });
      break;
    case 'scripted':
      title = _i18n.i18n.translate('indexPatternManagement.editIndexPattern.tabs.scriptedHeader', {
        defaultMessage: 'Scripted fields'
      });
      break;
    case 'sourceFilters':
      title = _i18n.i18n.translate('indexPatternManagement.editIndexPattern.tabs.sourceHeader', {
        defaultMessage: 'Field filters'
      });
      break;
  }
  const count = ` (${filteredCount[type] === totalCount[type] ? filteredCount[type] : filteredCount[type] + ' / ' + totalCount[type]})`;
  return title + count;
}
function getTabs(indexPattern, fieldFilter, relationshipCount = 0, scriptedFieldsEnabled) {
  const totalCount = getCounts(indexPattern.fields.getAll(), indexPattern.getSourceFiltering());
  const filteredCount = getCounts(indexPattern.fields.getAll(), indexPattern.getSourceFiltering(), fieldFilter);
  const tabs = [];
  tabs.push({
    name: getTitle('indexed', filteredCount, totalCount),
    id: _constants.TAB_INDEXED_FIELDS,
    'data-test-subj': 'tab-indexedFields'
  });
  if (!(0, _utils.isRollup)(indexPattern.type) && scriptedFieldsEnabled) {
    tabs.push({
      name: getTitle('scripted', filteredCount, totalCount),
      id: _constants.TAB_SCRIPTED_FIELDS,
      'data-test-subj': 'tab-scriptedFields'
    });
  }
  tabs.push({
    name: getTitle('sourceFilters', filteredCount, totalCount),
    id: _constants.TAB_SOURCE_FILTERS,
    'data-test-subj': 'tab-sourceFilters'
  });
  tabs.push({
    name: _i18n.i18n.translate('indexPatternManagement.editIndexPattern.tabs.relationshipsHeader', {
      defaultMessage: 'Relationships ({count})',
      values: {
        count: relationshipCount
      }
    }),
    id: _constants.TAB_RELATIONSHIPS,
    'data-test-subj': 'tab-relationships'
  });
  return tabs;
}
function getPath(field, indexPattern) {
  return `/dataView/${indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.id}/field/${encodeURIComponent(field.name)}`;
}
function convertToEuiFilterOptions(options) {
  return (0, _lodash.uniq)(options).map(option => {
    return {
      value: option,
      name: option
    };
  });
}