"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateIndexTemplate = void 0;
var _fp = require("lodash/fp");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Installs index template that uses installed component template
 * Prior to installation, simulates the installation to check for possible
 * conflicts. Simulate should return an empty mapping if a template
 * conflicts with an already installed template.
 */
const createOrUpdateIndexTemplate = async ({
  logger,
  esClient,
  template
}) => {
  logger.info(`Installing index template ${template.name}`);
  let mappings = {};
  try {
    // Simulate the index template to proactively identify any issues with the mapping
    const simulateResponse = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.simulateTemplate(template), {
      logger
    });
    mappings = simulateResponse.template.mappings;
  } catch (err) {
    logger.error(`Failed to simulate index template mappings for ${template.name}; not applying mappings - ${err.message}`, err);
    return;
  }
  if ((0, _fp.isEmpty)(mappings)) {
    throw new Error(`No mappings would be generated for ${template.name}, possibly due to failed/misconfigured bootstrapping`);
  }
  try {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.putIndexTemplate(template), {
      logger
    });
  } catch (err) {
    logger.error(`Error installing index template ${template.name} - ${err.message}`, err);
    throw err;
  }
};
exports.createOrUpdateIndexTemplate = createOrUpdateIndexTemplate;