"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flattenHit = flattenHit;
exports.tabifyDocs = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// meta fields we won't merge with our result hit
const EXCLUDED_META_FIELDS = ['_type', '_source'];

// This is an overwrite of the SearchHit type to add the ignored_field_values.
// Can be removed once the estypes.SearchHit knows about ignored_field_values

function flattenAccum(flat, obj, keyPrefix, indexPattern, params) {
  for (const k in obj) {
    if (!obj.hasOwnProperty(k)) {
      continue;
    }
    const val = obj[k];
    const key = keyPrefix + k;
    const field = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.fields.getByName(key);
    if ((params === null || params === void 0 ? void 0 : params.shallow) === false) {
      const isNestedField = (field === null || field === void 0 ? void 0 : field.type) === 'nested';
      if (Array.isArray(val) && !isNestedField) {
        for (let i = 0; i < val.length; i++) {
          const v = val[i];
          if ((0, _lodash.isPlainObject)(v)) {
            flattenAccum(flat, v, key + '.', indexPattern, params);
          }
        }
        continue;
      }
    } else if (flat[key] !== undefined) {
      continue;
    }
    const hasValidMapping = field && field.type !== 'conflict';
    const isValue = !(0, _lodash.isPlainObject)(val);
    if (hasValidMapping || isValue) {
      if (!flat[key]) {
        flat[key] = val;
      } else if (Array.isArray(flat[key])) {
        flat[key].push(val);
      } else {
        flat[key] = [flat[key], val];
      }
      continue;
    }
    flattenAccum(flat, val, key + '.', indexPattern, params);
  }
}

/**
 * Flattens an individual hit (from an ES response) into an object. This will
 * create flattened field names, like `user.name`.
 *
 * @param hit The hit from an ES reponse's hits.hits[]
 * @param indexPattern The index pattern for the requested index if available.
 * @param params Parameters how to flatten the hit
 */
function flattenHit(hit, indexPattern, params) {
  const flat = {};
  flattenAccum(flat, hit.fields || {}, '', indexPattern, params);
  if ((params === null || params === void 0 ? void 0 : params.source) !== false && hit._source) {
    flattenAccum(flat, hit._source, '', indexPattern, params);
  } else if (params !== null && params !== void 0 && params.includeIgnoredValues && hit.ignored_field_values) {
    // If enabled merge the ignored_field_values into the flattened hit. This will
    // merge values that are not actually indexed by ES (i.e. ignored), e.g. because
    // they were above the `ignore_above` limit or malformed for specific types.
    // This API will only contain the values that were actually ignored, i.e. for the same
    // field there might exist another value in the `fields` response, why this logic
    // merged them both together. We do not merge this (even if enabled) in case source has been
    // merged, since we would otherwise duplicate values, since ignore_field_values and _source
    // contain the same values.
    for (const fieldName in hit.ignored_field_values) {
      if (!hit.ignored_field_values.hasOwnProperty(fieldName)) {
        continue;
      }
      const fieldValue = hit.ignored_field_values[fieldName];
      if (flat[fieldName]) {
        // If there was already a value from the fields API, make sure we're merging both together
        if (Array.isArray(flat[fieldName])) {
          flat[fieldName] = [...flat[fieldName], ...fieldValue];
        } else {
          flat[fieldName] = [flat[fieldName], ...fieldValue];
        }
      } else {
        // If no previous value was assigned we can simply use the value from `ignored_field_values` as it is
        flat[fieldName] = fieldValue;
      }
    }
  }

  // Merge all valid meta fields into the flattened object
  if (indexPattern !== null && indexPattern !== void 0 && indexPattern.metaFields) {
    for (let i = 0; i < (indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.metaFields.length); i++) {
      const fieldName = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.metaFields[i];
      const isExcludedMetaField = EXCLUDED_META_FIELDS.includes(fieldName) || fieldName.charAt(0) !== '_';
      if (!isExcludedMetaField) {
        flat[fieldName] = hit[fieldName];
      }
    }
  }

  // Use a proxy to make sure that keys are always returned in a specific order,
  // so we have a guarantee on the flattened order of keys.
  return makeProxy(flat, indexPattern);
}
function makeProxy(flat, indexPattern) {
  function comparator(a, b) {
    var _indexPattern$metaFie, _indexPattern$metaFie2;
    const aIsMeta = indexPattern === null || indexPattern === void 0 ? void 0 : (_indexPattern$metaFie = indexPattern.metaFields) === null || _indexPattern$metaFie === void 0 ? void 0 : _indexPattern$metaFie.includes(String(a));
    const bIsMeta = indexPattern === null || indexPattern === void 0 ? void 0 : (_indexPattern$metaFie2 = indexPattern.metaFields) === null || _indexPattern$metaFie2 === void 0 ? void 0 : _indexPattern$metaFie2.includes(String(b));
    if (aIsMeta && bIsMeta) {
      return String(a).localeCompare(String(b));
    }
    if (aIsMeta) {
      return 1;
    }
    if (bIsMeta) {
      return -1;
    }
    return String(a).localeCompare(String(b));
  }
  return new Proxy(flat, {
    ownKeys: target => {
      return Reflect.ownKeys(target).sort(comparator);
    }
  });
}
const tabifyDocs = (esResponse, index, params = {}) => {
  const columns = [];
  const rows = esResponse.hits.hits.map(hit => {
    const flat = flattenHit(hit, index, params);
    if (!flat) {
      return;
    }
    for (const [key, value] of Object.entries(flat)) {
      const field = index === null || index === void 0 ? void 0 : index.fields.getByName(key);
      const fieldName = (field === null || field === void 0 ? void 0 : field.name) || key;
      if (!columns.find(c => c.id === fieldName)) {
        const fieldType = (field === null || field === void 0 ? void 0 : field.type) || typeof value;
        const formatter = field && (index === null || index === void 0 ? void 0 : index.getFormatterForField(field));
        columns.push({
          id: fieldName,
          name: fieldName,
          meta: {
            type: fieldType,
            field: fieldName,
            index: index === null || index === void 0 ? void 0 : index.id,
            params: formatter ? formatter.toJSON() : undefined
          }
        });
      }
    }
    return flat;
  }).filter(hit => hit);
  return {
    type: 'datatable',
    columns,
    rows: rows
  };
};
exports.tabifyDocs = tabifyDocs;