"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esqlAsyncSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _async_utils = require("../common/async_utils");
var _common = require("../../../../common");
var _report_search_error = require("../../report_search_error");
var _response_utils = require("./response_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// `drop_null_columns` is going to change the response
// now we get `all_columns` and `columns`
// `columns` contain only columns with data
// `all_columns` contain everything

const esqlAsyncSearchStrategyProvider = (searchConfig, logger, useInternalUser = false) => {
  function cancelAsyncSearch(id, esClient) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    return client.transport.request({
      method: 'DELETE',
      path: `/_query/async/${id}`
    }, {
      meta: true,
      // we don't want the ES client to retry (default value is 3)
      maxRetries: 0
    });
  }
  function asyncSearch({
    id,
    ...request
  }, options, {
    esClient,
    uiSettingsClient
  }) {
    var _request$params;
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const {
      dropNullColumns,
      ...requestParams
    } = (_request$params = request.params) !== null && _request$params !== void 0 ? _request$params : {};
    const search = async () => {
      var _request$params2, _request$params3, _meta$request;
      const params = id ? {
        ...(0, _async_utils.getCommonDefaultAsyncGetParams)(searchConfig, options),
        ...((_request$params2 = request.params) !== null && _request$params2 !== void 0 && _request$params2.keep_alive ? {
          keep_alive: request.params.keep_alive
        } : {}),
        ...((_request$params3 = request.params) !== null && _request$params3 !== void 0 && _request$params3.wait_for_completion_timeout ? {
          wait_for_completion_timeout: request.params.wait_for_completion_timeout
        } : {})
      } : {
        ...(await (0, _async_utils.getCommonDefaultAsyncSubmitParams)(searchConfig, options)),
        ...requestParams
      };
      const {
        body,
        headers,
        meta
      } = id ? await client.transport.request({
        method: 'GET',
        path: `/_query/async/${id}`,
        querystring: {
          ...params
        }
      }, {
        ...options.transport,
        signal: options.abortSignal,
        meta: true
      }) : await client.transport.request({
        method: 'POST',
        path: `/_query/async`,
        body: params,
        querystring: dropNullColumns ? 'drop_null_columns' : ''
      }, {
        ...options.transport,
        signal: options.abortSignal,
        meta: true
      });
      const finalResponse = (0, _response_utils.toAsyncKibanaSearchResponse)(body, headers === null || headers === void 0 ? void 0 : headers.warning,
      // do not return requestParams on polling calls
      id ? undefined : meta === null || meta === void 0 ? void 0 : (_meta$request = meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
      return finalResponse;
    };
    const cancel = async () => {
      if (!id || options.isStored) return;
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        var _e$meta;
        // A 404 means either this search request does not exist, or that it is already cancelled
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

        // Log all other (unexpected) error messages
        logger.error(`cancelEsqlAsyncSearch error: ${e.message}`);
      }
    };
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: searchConfig.asyncSearch.pollInterval,
      ...options
    }).pipe((0, _rxjs.tap)(response => id = response.id), (0, _rxjs.catchError)(e => {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }));
  }
  return {
    /**
     * @param request
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Observable<IKibanaResponse<SqlGetAsyncResponse>>`
     * @throws `KbnSearchError`
     */
    search: (request, options, deps) => {
      logger.debug(`search ${JSON.stringify(request) || request.id}`);
      return asyncSearch(request, options, deps);
    },
    /**
     * @param id async search ID to cancel, as returned from _async_search API
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    cancel: async (id, options, {
      esClient
    }) => {
      logger.debug(`cancel ${id}`);
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    /**
     *
     * @param id async search ID to extend, as returned from _async_search API
     * @param keepAlive
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`extend ${id} by ${keepAlive}`);
      try {
        const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
        await client.transport.request({
          method: 'GET',
          path: `/_query/async/${id}`,
          body: {
            id,
            keep_alive: keepAlive
          }
        }, {
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        });
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    }
  };
};
exports.esqlAsyncSearchStrategyProvider = esqlAsyncSearchStrategyProvider;