"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.INITIAL_SEARCH_SESSION_REST_VERSION = void 0;
exports.registerSessionRoutes = registerSessionRoutes;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/kibana-utils-plugin/server");
var _response_schema = require("./response_schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const STORE_SEARCH_SESSIONS_ROLE_TAG = `access:store_search_session`;
const access = 'internal';
const options = {
  tags: [STORE_SEARCH_SESSIONS_ROLE_TAG]
};
const pathPrefix = '/internal/session';
const INITIAL_SEARCH_SESSION_REST_VERSION = exports.INITIAL_SEARCH_SESSION_REST_VERSION = '1';
const version = INITIAL_SEARCH_SESSION_REST_VERSION;
const idAndAttrsOnly = so => so && {
  id: so.id,
  attributes: so.attributes
};
function registerSessionRoutes(router, logger) {
  router.versioned.post({
    path: pathPrefix,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        body: _configSchema.schema.object({
          sessionId: _configSchema.schema.string(),
          name: _configSchema.schema.string(),
          appId: _configSchema.schema.string(),
          expires: _configSchema.schema.maybe(_configSchema.schema.string()),
          locatorId: _configSchema.schema.string(),
          initialState: _configSchema.schema.maybe(_configSchema.schema.object({}, {
            unknowns: 'allow'
          })),
          restoreState: _configSchema.schema.maybe(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))
        })
      },
      response: {
        200: {
          body: () => _configSchema.schema.maybe((0, _response_schema.searchSessionSchema)())
        }
      }
    }
  }, async (context, request, res) => {
    const {
      sessionId,
      name,
      expires,
      initialState,
      restoreState,
      appId,
      locatorId
    } = request.body;
    try {
      const searchContext = await context.search;
      const response = await searchContext.saveSession(sessionId, {
        name,
        appId,
        expires,
        locatorId,
        initialState,
        restoreState
      });
      const body = idAndAttrsOnly(response);
      return res.ok({
        body
      });
    } catch (err) {
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.get({
    path: `${pathPrefix}/{id}`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      },
      response: {
        200: {
          body: _response_schema.searchSessionSchema
        }
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    try {
      const searchContext = await context.search;
      const response = await searchContext.getSession(id);
      const body = idAndAttrsOnly(response);
      return res.ok({
        body
      });
    } catch (e) {
      var _e$output;
      const err = ((_e$output = e.output) === null || _e$output === void 0 ? void 0 : _e$output.payload) || e;
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.get({
    path: `${pathPrefix}/{id}/status`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      },
      response: {
        200: {
          body: _response_schema.searchSessionStatusSchema
        }
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    try {
      const searchContext = await context.search;
      const response = await searchContext.getSessionStatus(id);
      return res.ok({
        body: response
      });
    } catch (e) {
      var _e$output2;
      const err = ((_e$output2 = e.output) === null || _e$output2 === void 0 ? void 0 : _e$output2.payload) || e;
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.post({
    path: `${pathPrefix}/_find`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        body: _configSchema.schema.object({
          page: _configSchema.schema.maybe(_configSchema.schema.number()),
          perPage: _configSchema.schema.maybe(_configSchema.schema.number()),
          sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
          sortOrder: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('desc'), _configSchema.schema.literal('asc')])),
          filter: _configSchema.schema.maybe(_configSchema.schema.string()),
          searchFields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          search: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      },
      response: {
        200: {
          body: _response_schema.searchSessionsFindSchema
        }
      }
    }
  }, async (context, request, res) => {
    const {
      page,
      perPage,
      sortField,
      sortOrder,
      filter,
      searchFields,
      search
    } = request.body;
    try {
      const searchContext = await context.search;
      const response = await searchContext.findSessions({
        page,
        perPage,
        sortField,
        sortOrder,
        filter,
        searchFields,
        search
      });
      const body = {
        total: response.total,
        saved_objects: response.saved_objects.map(idAndAttrsOnly),
        statuses: response.statuses
      };
      return res.ok({
        body
      });
    } catch (err) {
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.delete({
    path: `${pathPrefix}/{id}`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    try {
      const searchContext = await context.search;
      await searchContext.deleteSession(id);
      return res.ok();
    } catch (e) {
      var _e$output3;
      const err = ((_e$output3 = e.output) === null || _e$output3 === void 0 ? void 0 : _e$output3.payload) || e;
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.post({
    path: `${pathPrefix}/{id}/cancel`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    try {
      const searchContext = await context.search;
      await searchContext.cancelSession(id);
      return res.ok();
    } catch (e) {
      var _e$output4;
      const err = ((_e$output4 = e.output) === null || _e$output4 === void 0 ? void 0 : _e$output4.payload) || e;
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.put({
    path: `${pathPrefix}/{id}`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        }),
        body: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string()),
          expires: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      },
      response: {
        200: {
          body: _response_schema.searchSessionsUpdateSchema
        }
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    const {
      name,
      expires
    } = request.body;
    try {
      const searchContext = await context.search;
      const response = await searchContext.updateSession(id, {
        name,
        expires
      });
      return res.ok({
        body: response
      });
    } catch (err) {
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
  router.versioned.post({
    path: `${pathPrefix}/{id}/_extend`,
    access,
    options
  }).addVersion({
    version,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        }),
        body: _configSchema.schema.object({
          expires: _configSchema.schema.string()
        })
      },
      response: {
        200: {
          body: _response_schema.searchSessionsUpdateSchema
        }
      }
    }
  }, async (context, request, res) => {
    const {
      id
    } = request.params;
    const {
      expires
    } = request.body;
    try {
      const searchContext = await context.search;
      const response = await searchContext.extendSession(id, new Date(expires));
      return res.ok({
        body: response
      });
    } catch (e) {
      var _e$output5;
      const err = ((_e$output5 = e.output) === null || _e$output5 === void 0 ? void 0 : _e$output5.payload) || e;
      logger.error(err);
      return (0, _server.reportServerError)(res, err);
    }
  });
}