"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateFilters = generateFilters;
var _lodash = _interopRequireDefault(require("lodash"));
var _esQuery = require("@kbn/es-query");
var _fieldTypes = require("@kbn/field-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getExistingFilter(appFilters, fieldName, value) {
  // TODO: On array fields, negating does not negate the combination, rather all terms
  return _lodash.default.find(appFilters, function (filter) {
    if (!filter) return;
    if (fieldName === '_exists_' && (0, _esQuery.isExistsFilter)(filter)) {
      return filter.query.exists.field === value;
    }
    if ((0, _esQuery.isPhraseFilter)(filter)) {
      return (0, _esQuery.getPhraseFilterField)(filter) === fieldName && (0, _esQuery.getPhraseFilterValue)(filter) === value;
    }
    if ((0, _esQuery.isScriptedPhraseFilter)(filter)) {
      var _filter$query, _filter$query$script, _filter$query$script$, _filter$query$script$2;
      return filter.meta.field === fieldName && ((_filter$query = filter.query) === null || _filter$query === void 0 ? void 0 : (_filter$query$script = _filter$query.script) === null || _filter$query$script === void 0 ? void 0 : (_filter$query$script$ = _filter$query$script.script) === null || _filter$query$script$ === void 0 ? void 0 : (_filter$query$script$2 = _filter$query$script$.params) === null || _filter$query$script$2 === void 0 ? void 0 : _filter$query$script$2.value) === value;
    }
    if ((0, _esQuery.isRangeFilter)(filter)) {
      return (0, _esQuery.getFilterField)(filter) === fieldName && _lodash.default.isEqual(filter.query.range[fieldName], value);
    }
  });
}
function updateExistingFilter(existingFilter, negate) {
  existingFilter.meta.disabled = false;
  if (existingFilter.meta.negate !== negate) {
    existingFilter.meta.negate = !existingFilter.meta.negate;
  }
}

/**
 * Generate filter objects, as a result of triggering a filter action on a
 * specific index pattern field.
 *
 * @param {FilterManager} filterManager - The active filter manager to lookup for existing filters
 * @param {Field | string} field - The field for which filters should be generated
 * @param {any} values - One or more values to filter for.
 * @param {string} operation - "-" to create a negated filter
 * @param {string} index - Index string to generate filters for
 *
 * @returns {object} An array of filters to be added back to filterManager
 */
function generateFilters(filterManager, field, values, operation, index) {
  values = Array.isArray(values) ? _lodash.default.uniq(values) : [values];
  const fieldObj = _lodash.default.isObject(field) ? field : {
    name: field
  };
  const fieldName = fieldObj.name;
  const appFilters = filterManager.getAppFilters();
  const negate = operation === '-';
  function generateFilter(value) {
    var _fieldObj$type;
    const isRange = ((_fieldObj$type = fieldObj.type) === null || _fieldObj$type === void 0 ? void 0 : _fieldObj$type.includes('range')) || fieldObj.type === _fieldTypes.KBN_FIELD_TYPES.DATE;
    if (isRange && _lodash.default.isObjectLike(value)) {
      return (0, _esQuery.buildFilter)(index, fieldObj, _esQuery.FILTERS.RANGE_FROM_VALUE, negate, false, value, null, _esQuery.FilterStateStore.APP_STATE);
    }

    // exists filter special case:  fieldname = '_exists' and value = fieldname
    const filterType = fieldName === '_exists_' ? _esQuery.FILTERS.EXISTS : _esQuery.FILTERS.PHRASE;
    const actualFieldObj = fieldName === '_exists_' ? {
      name: value
    } : fieldObj;

    // Fix for #7189 - if value is empty, phrase filters become exists filters.
    const isNullFilter = value === null || value === undefined;
    return (0, _esQuery.buildFilter)(index, actualFieldObj, isNullFilter ? _esQuery.FILTERS.EXISTS : filterType, isNullFilter ? !negate : negate, false, value, null, _esQuery.FilterStateStore.APP_STATE);
  }

  /**
   * When filtering on a date, instead of simply creating a "match_phrase" or "match" query (which isn't useful when
   * specific date formats are involved), we create a range query that only includes this date.
   * NOTE: This assumes that the value passed in is already in an appropriate format (such as date_time or
   * strict_date_optional_time_nanos).
   * @param value
   */
  function castValue(value) {
    if (fieldObj.type === _fieldTypes.KBN_FIELD_TYPES.DATE && typeof value === 'string') {
      var _fieldObj$esTypes;
      const format = (_fieldObj$esTypes = fieldObj.esTypes) !== null && _fieldObj$esTypes !== void 0 && _fieldObj$esTypes.includes('date_nanos') ? 'strict_date_optional_time_nanos' : 'date_time';
      return {
        format,
        gte: value,
        lte: value
      };
    }
    return value;
  }
  return _lodash.default.chain(values).map(castValue).map(value => {
    const existing = getExistingFilter(appFilters, fieldName, value);
    if (existing) {
      updateExistingFilter(existing, negate);
    }
    return existing !== null && existing !== void 0 ? existing : generateFilter(value);
  }).value();
}