"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchSource = void 0;
var _common = require("@kbn/data-views-plugin/common");
var _migrate_legacy_query = require("./migrate_legacy_query");
var _search_source = require("./search_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Deserializes a json string and a set of referenced objects to a `SearchSource` instance.
 * Use this method to re-create the search source serialized using `searchSource.serialize`.
 *
 * This function is a factory function that returns the actual utility when calling it with the
 * required service dependency (index patterns contract). A pre-wired version is also exposed in
 * the start contract of the data plugin as part of the search service
 *
 * @param indexPatterns The index patterns contract of the data plugin
 * @param searchSourceDependencies
 *
 * @return Wired utility function taking two parameters `searchSourceJson`, the json string
 * returned by `serializeSearchSource` and `references`, a list of references including the ones
 * returned by `serializeSearchSource`.
 *
 *
 * @public */
const createSearchSource = (indexPatterns, searchSourceDependencies) => {
  let dataViewLazy;
  const createFields = async (searchSourceFields = {}, useDataViewLazy = false) => {
    const {
      index,
      parent,
      ...restOfFields
    } = searchSourceFields;
    const fields = {
      ...restOfFields
    };

    // hydrating index pattern
    if (searchSourceFields.index) {
      if (!useDataViewLazy) {
        fields.index = typeof searchSourceFields.index === 'string' ? await indexPatterns.get(searchSourceFields.index) : await indexPatterns.create(searchSourceFields.index);
      } else {
        dataViewLazy = typeof searchSourceFields.index === 'string' ? await indexPatterns.getDataViewLazy(searchSourceFields.index) : await indexPatterns.createDataViewLazy(searchSourceFields.index);
        const [spec, shortDotsEnable, metaFields] = await Promise.all([dataViewLazy.toSpec(), searchSourceDependencies.dataViews.getShortDotsEnable(), searchSourceDependencies.dataViews.getMetaFields()]);
        const dataView = new _common.DataView({
          spec,
          // field format functionality is not used within search source
          fieldFormats: {},
          shortDotsEnable,
          metaFields
        });
        fields.index = dataView;
      }
    }
    if (searchSourceFields.parent) {
      fields.parent = await createFields(searchSourceFields.parent);
    }
    return fields;
  };
  const createSearchSourceFn = async (searchSourceFields = {}, useDataViewLazy) => {
    const fields = await createFields(searchSourceFields, !!useDataViewLazy);
    const searchSource = new _search_source.SearchSource(fields, searchSourceDependencies);

    // todo: move to migration script .. create issue
    const query = searchSource.getOwnField('query');
    if (typeof query !== 'undefined') {
      searchSource.setField('query', (0, _migrate_legacy_query.migrateLegacyQuery)(query));
    }
    // using the dataViewLazy check as a type guard
    if (useDataViewLazy && dataViewLazy) {
      var _fields$index;
      const dataViewFields = await searchSource.loadDataViewFields(dataViewLazy);
      (_fields$index = fields.index) === null || _fields$index === void 0 ? void 0 : _fields$index.fields.replaceAll(Object.values(dataViewFields).map(fld => fld.toSpec()));
    }
    return searchSource;
  };
  return createSearchSourceFn;
};
exports.createSearchSource = createSearchSource;