"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findDashboardById = findDashboardById;
exports.findDashboardIdByTitle = findDashboardIdByTitle;
exports.findDashboardsByIds = findDashboardsByIds;
exports.searchDashboards = searchDashboards;
var _dashboard_constants = require("../../../dashboard_constants");
var _dashboard_content_management_service = require("../dashboard_content_management_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

async function searchDashboards({
  contentManagement,
  hasNoReference,
  hasReference,
  options,
  search,
  size
}) {
  const {
    hits,
    pagination: {
      total
    }
  } = await contentManagement.client.search({
    contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
    query: {
      text: search ? `${search}*` : undefined,
      limit: size,
      tags: {
        included: (hasReference !== null && hasReference !== void 0 ? hasReference : []).map(({
          id
        }) => id),
        excluded: (hasNoReference !== null && hasNoReference !== void 0 ? hasNoReference : []).map(({
          id
        }) => id)
      }
    },
    options
  });
  return {
    total,
    hits
  };
}
async function findDashboardById(contentManagement, id) {
  /** If the dashboard exists in the cache, then return the result from that */
  const cachedDashboard = _dashboard_content_management_service.dashboardContentManagementCache.fetchDashboard(id);
  if (cachedDashboard) {
    return {
      id,
      status: 'success',
      attributes: cachedDashboard.item.attributes,
      references: cachedDashboard.item.references
    };
  }

  /** Otherwise, fetch the dashboard from the content management client, add it to the cache, and return the result */
  try {
    const response = await contentManagement.client.get({
      contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
      id
    });
    if (response.item.error) {
      throw response.item.error;
    }
    _dashboard_content_management_service.dashboardContentManagementCache.addDashboard(response);
    return {
      id,
      status: 'success',
      attributes: response.item.attributes,
      references: response.item.references
    };
  } catch (e) {
    return {
      status: 'error',
      error: e.body || e.message,
      id
    };
  }
}
async function findDashboardsByIds(contentManagement, ids) {
  const findPromises = ids.map(id => findDashboardById(contentManagement, id));
  const results = await Promise.all(findPromises);
  return results;
}
async function findDashboardIdByTitle(contentManagement, title) {
  const {
    hits
  } = await contentManagement.client.search({
    contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
    query: {
      text: title ? `${title}*` : undefined,
      limit: 10
    },
    options: {
      onlyTitle: true
    }
  });
  // The search isn't an exact match, lets see if we can find a single exact match to use
  const matchingDashboards = hits.filter(hit => hit.attributes.title.toLowerCase() === title.toLowerCase());
  if (matchingDashboards.length === 1) {
    return {
      id: matchingDashboards[0].id
    };
  }
}