"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDashboardUnsavedChanges = getDashboardUnsavedChanges;
exports.getDiffingMiddleware = getDiffingMiddleware;
exports.reducersToIgnore = exports.keysNotConsideredUnsavedChanges = void 0;
exports.startDiffingDashboardState = startDiffingDashboardState;
var _presentationPublishing = require("@kbn/presentation-publishing");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _dashboard_constants = require("../../../dashboard_constants");
var _plugin_services = require("../../../services/plugin_services");
var _dashboard_diffing_functions = require("./dashboard_diffing_functions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * An array of reducers which cannot cause unsaved changes. Unsaved changes only compares the explicit input
 * and the last saved input, so we can safely ignore any output reducers, and most componentState reducers.
 * This is only for performance reasons, because the diffing function itself can be quite heavy.
 */
const reducersToIgnore = exports.reducersToIgnore = ['setTimeslice', 'setFullScreenMode', 'setExpandedPanelId', 'setHasUnsavedChanges'];

/**
 * Some keys will often have deviated from their last saved state, but should not persist over reloads
 */
const keysToOmitFromSessionStorage = ['lastReloadRequestTime', 'executionContext', 'timeslice', 'id', 'timeRange',
// Current behaviour expects time range not to be backed up. Revisit this?
'refreshInterval'];

/**
 * Some keys will often have deviated from their last saved state, but should be
 * ignored when calculating whether or not this dashboard has unsaved changes.
 */
const keysNotConsideredUnsavedChanges = exports.keysNotConsideredUnsavedChanges = ['lastReloadRequestTime', 'executionContext', 'timeslice', 'viewMode', 'id'];

/**
 * build middleware that fires an event any time a reducer that could cause unsaved changes is run
 */
function getDiffingMiddleware() {
  const diffingMiddleware = store => next => action => {
    var _action$type$split;
    const dispatchedActionName = (_action$type$split = action.type.split('/')) === null || _action$type$split === void 0 ? void 0 : _action$type$split[1];
    if (dispatchedActionName && dispatchedActionName !== 'updateEmbeddableReduxOutput' &&
    // ignore any generic output updates.
    !reducersToIgnore.includes(dispatchedActionName)) {
      this.anyReducerRun.next(null);
    }
    next(action);
  };
  return diffingMiddleware;
}

/**
 * Does an initial diff between @param initialInput and @param initialLastSavedInput, and creates a middleware
 * which listens to the redux store and pushes updates to the `hasUnsavedChanges` and `backupUnsavedChanges` behaviour
 * subjects so that the corresponding subscriptions can dispatch updates as necessary
 */
function startDiffingDashboardState(creationOptions) {
  var _this$controlGroup$un, _this$controlGroup;
  /**
   *  Create an observable stream of unsaved changes from all react embeddable children
   */
  const reactEmbeddableUnsavedChanges = this.children$.pipe((0, _rxjs.map)(children => Object.keys(children)), (0, _rxjs.distinctUntilChanged)(_fastDeepEqual.default),
  // children may change, so make sure we subscribe/unsubscribe with switchMap
  (0, _rxjs.switchMap)(newChildIds => {
    if (newChildIds.length === 0) return (0, _rxjs.of)([]);
    const childrenThatPublishUnsavedChanges = Object.entries(this.children$.value).filter(([childId, child]) => (0, _presentationPublishing.apiPublishesUnsavedChanges)(child));
    if (childrenThatPublishUnsavedChanges.length === 0) return (0, _rxjs.of)([]);
    return (0, _rxjs.combineLatest)(childrenThatPublishUnsavedChanges.map(([childId, child]) => child.unsavedChanges.pipe((0, _rxjs.map)(unsavedChanges => ({
      childId,
      unsavedChanges
    })))));
  }), (0, _rxjs.debounceTime)(_dashboard_constants.CHANGE_CHECK_DEBOUNCE), (0, _rxjs.map)(children => children.filter(child => Boolean(child.unsavedChanges))));

  /**
   * Create an observable stream that checks for unsaved changes in the Dashboard state
   * and the state of all of its legacy embeddable children.
   */
  const dashboardUnsavedChanges = this.anyReducerRun.pipe((0, _rxjs.startWith)(null), (0, _rxjs.debounceTime)(_dashboard_constants.CHANGE_CHECK_DEBOUNCE), (0, _rxjs.switchMap)(() => {
    return (async () => {
      const {
        explicitInput: currentInput,
        componentState: {
          lastSavedInput
        }
      } = this.getState();
      const unsavedChanges = await getDashboardUnsavedChanges.bind(this)(lastSavedInput, currentInput);
      return unsavedChanges;
    })();
  }));

  /**
   * Combine unsaved changes from all sources together. Set unsaved changes state and backup unsaved changes when any of the sources emit.
   */
  this.diffingSubscription.add((0, _rxjs.combineLatest)([dashboardUnsavedChanges, reactEmbeddableUnsavedChanges, (_this$controlGroup$un = (_this$controlGroup = this.controlGroup) === null || _this$controlGroup === void 0 ? void 0 : _this$controlGroup.unsavedChanges) !== null && _this$controlGroup$un !== void 0 ? _this$controlGroup$un : (0, _rxjs.of)(undefined)]).subscribe(([dashboardChanges, reactEmbeddableChanges, controlGroupChanges]) => {
    // calculate unsaved changes
    const hasUnsavedChanges = Object.keys((0, _lodash.omit)(dashboardChanges, keysNotConsideredUnsavedChanges)).length > 0 || reactEmbeddableChanges.length > 0 || controlGroupChanges !== undefined;
    if (hasUnsavedChanges !== this.getState().componentState.hasUnsavedChanges) {
      this.dispatch.setHasUnsavedChanges(hasUnsavedChanges);
    }
    const unsavedPanelState = reactEmbeddableChanges.reduce((acc, {
      childId,
      unsavedChanges
    }) => {
      acc[childId] = unsavedChanges;
      return acc;
    }, {});

    // backup unsaved changes if configured to do so
    if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useSessionStorageIntegration) {
      backupUnsavedChanges.bind(this)(dashboardChanges, unsavedPanelState, controlGroupChanges);
    }
  }));
}

/**
 * Does a shallow diff between @param lastInput and @param input and
 * @returns an object out of the keys which are different.
 */
async function getDashboardUnsavedChanges(lastInput, input) {
  const allKeys = [...new Set([...Object.keys(lastInput), ...Object.keys(input)])];
  const keyComparePromises = allKeys.map(key => new Promise(resolve => {
    if (input[key] === undefined && lastInput[key] === undefined) {
      resolve({
        key,
        isEqual: true
      });
    }
    (0, _dashboard_diffing_functions.isKeyEqualAsync)(key, {
      container: this,
      currentValue: input[key],
      currentInput: input,
      lastValue: lastInput[key],
      lastInput
    }, _dashboard_diffing_functions.unsavedChangesDiffingFunctions).then(isEqual => resolve({
      key,
      isEqual
    }));
  }));
  const inputChanges = (await Promise.allSettled(keyComparePromises)).reduce((changes, current) => {
    if (current.status === 'fulfilled') {
      const {
        key,
        isEqual
      } = current.value;
      if (!isEqual) changes[key] = input[key];
    }
    return changes;
  }, {});
  return inputChanges;
}
function backupUnsavedChanges(dashboardChanges, reactEmbeddableChanges, controlGroupChanges) {
  const {
    dashboardBackup
  } = _plugin_services.pluginServices.getServices();
  const dashboardStateToBackup = (0, _lodash.omit)(dashboardChanges, keysToOmitFromSessionStorage);
  dashboardBackup.setState(this.getDashboardSavedObjectId(), {
    ...dashboardStateToBackup,
    panels: dashboardChanges.panels,
    controlGroupInput: controlGroupChanges
  }, reactEmbeddableChanges);
}