"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runInteractiveSave = runInteractiveSave;
exports.runQuickSave = runQuickSave;
var _ebtTools = require("@kbn/ebt-tools");
var _public = require("@kbn/embeddable-plugin/public");
var _presentationContainers = require("@kbn/presentation-containers");
var _public2 = require("@kbn/saved-objects-plugin/public");
var _lodash = require("lodash");
var _react = _interopRequireDefault(require("react"));
var _reactRedux = require("react-redux");
var _i18n = require("@kbn/i18n");
var _common = require("../../../../common");
var _dashboard_constants = require("../../../dashboard_constants");
var _plugin_services = require("../../../services/plugin_services");
var _extract_title_and_count = require("./lib/extract_title_and_count");
var _save_modal = require("./overlays/save_modal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const serializeAllPanelState = async dashboard => {
  const {
    embeddable: {
      reactEmbeddableRegistryHasKey
    }
  } = _plugin_services.pluginServices.getServices();
  const references = [];
  const panels = (0, _lodash.cloneDeep)(dashboard.getInput().panels);
  const serializePromises = [];
  for (const [uuid, panel] of Object.entries(panels)) {
    if (!reactEmbeddableRegistryHasKey(panel.type)) continue;
    const api = dashboard.children$.value[uuid];
    if (api && (0, _presentationContainers.apiHasSerializableState)(api)) {
      serializePromises.push((async () => {
        const serialized = await api.serializeState();
        return {
          uuid,
          serialized
        };
      })());
    }
  }
  const serializeResults = await Promise.all(serializePromises);
  for (const result of serializeResults) {
    var _result$serialized$re;
    panels[result.uuid].explicitInput = {
      ...result.serialized.rawState,
      id: result.uuid
    };
    references.push(...(0, _common.prefixReferencesFromPanel)(result.uuid, (_result$serialized$re = result.serialized.references) !== null && _result$serialized$re !== void 0 ? _result$serialized$re : []));
  }
  return {
    panels,
    references
  };
};

/**
 * Save the current state of this dashboard to a saved object without showing any save modal.
 */
async function runQuickSave() {
  var _saveResult$reference;
  const {
    dashboardContentManagement: {
      saveDashboardState
    }
  } = _plugin_services.pluginServices.getServices();
  const {
    explicitInput: currentState,
    componentState: {
      lastSavedId,
      managed
    }
  } = this.getState();
  if (managed) return;
  const {
    panels: nextPanels,
    references
  } = await serializeAllPanelState(this);
  const dashboardStateToSave = {
    ...currentState,
    panels: nextPanels
  };
  let stateToSave = dashboardStateToSave;
  let persistableControlGroupInput;
  if (this.controlGroup) {
    persistableControlGroupInput = this.controlGroup.getPersistableInput();
    stateToSave = {
      ...stateToSave,
      controlGroupInput: persistableControlGroupInput
    };
  }
  const saveResult = await saveDashboardState({
    panelReferences: references,
    currentState: stateToSave,
    saveOptions: {},
    lastSavedId
  });
  this.savedObjectReferences = (_saveResult$reference = saveResult.references) !== null && _saveResult$reference !== void 0 ? _saveResult$reference : [];
  this.dispatch.setLastSavedInput(dashboardStateToSave);
  this.saveNotification$.next();
  if (this.controlGroup && persistableControlGroupInput) {
    this.controlGroup.setSavedState(persistableControlGroupInput);
  }
  return saveResult;
}

/**
 * @description exclusively for user directed dashboard save actions, also
 * accounts for scenarios of cloning elastic managed dashboard into user managed dashboards
 */
async function runInteractiveSave(interactionMode) {
  const {
    data: {
      query: {
        timefilter: {
          timefilter
        }
      }
    },
    savedObjectsTagging: {
      hasApi: hasSavedObjectsTagging
    },
    dashboardContentManagement: {
      checkForDuplicateDashboardTitle,
      saveDashboardState
    }
  } = _plugin_services.pluginServices.getServices();
  const {
    explicitInput: currentState,
    componentState: {
      lastSavedId,
      managed
    }
  } = this.getState();
  return new Promise((resolve, reject) => {
    if (interactionMode === _public.ViewMode.EDIT && managed) {
      resolve(undefined);
    }
    const onSaveAttempt = async ({
      newTags,
      newTitle,
      newDescription,
      newCopyOnSave,
      newTimeRestore,
      onTitleDuplicate,
      isTitleDuplicateConfirmed
    }) => {
      const saveOptions = {
        confirmOverwrite: false,
        isTitleDuplicateConfirmed,
        onTitleDuplicate,
        saveAsCopy: lastSavedId ? true : newCopyOnSave
      };
      try {
        var _saveResult$reference2;
        if (!(await checkForDuplicateDashboardTitle({
          title: newTitle,
          onTitleDuplicate,
          lastSavedTitle: currentState.title,
          copyOnSave: saveOptions.saveAsCopy,
          isTitleDuplicateConfirmed
        }))) {
          return {};
        }
        const stateFromSaveModal = {
          title: newTitle,
          tags: [],
          description: newDescription,
          timeRestore: newTimeRestore,
          timeRange: newTimeRestore ? timefilter.getTime() : undefined,
          refreshInterval: newTimeRestore ? timefilter.getRefreshInterval() : undefined
        };
        if (hasSavedObjectsTagging && newTags) {
          // remove `hasSavedObjectsTagging` once the savedObjectsTagging service is optional
          stateFromSaveModal.tags = newTags;
        }
        let dashboardStateToSave = {
          ...currentState,
          ...stateFromSaveModal
        };
        let persistableControlGroupInput;
        if (this.controlGroup) {
          persistableControlGroupInput = this.controlGroup.getPersistableInput();
          dashboardStateToSave = {
            ...dashboardStateToSave,
            controlGroupInput: persistableControlGroupInput
          };
        }
        const {
          panels: nextPanels,
          references
        } = await serializeAllPanelState(this);
        const newPanels = await (async () => {
          if (!managed) return nextPanels;

          // this is a managed dashboard - unlink all by reference embeddables on clone
          const unlinkedPanels = {};
          for (const [panelId, panel] of Object.entries(nextPanels)) {
            const child = this.getChild(panelId);
            if (child && (0, _public.isReferenceOrValueEmbeddable)(child) && child.inputIsRefType(child.getInput())) {
              const valueTypeInput = await child.getInputAsValueType();
              unlinkedPanels[panelId] = {
                ...panel,
                explicitInput: valueTypeInput
              };
              continue;
            }
            unlinkedPanels[panelId] = panel;
          }
          return unlinkedPanels;
        })();
        const beforeAddTime = window.performance.now();
        const saveResult = await saveDashboardState({
          panelReferences: references,
          saveOptions,
          currentState: {
            ...dashboardStateToSave,
            panels: newPanels,
            title: newTitle
          },
          lastSavedId
        });
        const addDuration = window.performance.now() - beforeAddTime;
        (0, _ebtTools.reportPerformanceMetricEvent)(_plugin_services.pluginServices.getServices().analytics, {
          eventName: _dashboard_constants.SAVED_OBJECT_POST_TIME,
          duration: addDuration,
          meta: {
            saved_object_type: _dashboard_constants.DASHBOARD_CONTENT_ID
          }
        });
        stateFromSaveModal.lastSavedId = saveResult.id;
        if (saveResult.id) {
          (0, _reactRedux.batch)(() => {
            this.dispatch.setStateFromSaveModal(stateFromSaveModal);
            this.dispatch.setLastSavedInput(dashboardStateToSave);
            if (this.controlGroup && persistableControlGroupInput) {
              this.controlGroup.setSavedState(persistableControlGroupInput);
            }
          });
        }
        this.savedObjectReferences = (_saveResult$reference2 = saveResult.references) !== null && _saveResult$reference2 !== void 0 ? _saveResult$reference2 : [];
        this.saveNotification$.next();
        resolve(saveResult);
        return saveResult;
      } catch (error) {
        reject(error);
        return error;
      }
    };
    (async () => {
      try {
        var _currentState$descrip;
        let customModalTitle;
        let newTitle = currentState.title;
        if (lastSavedId) {
          const [baseTitle, baseCount] = (0, _extract_title_and_count.extractTitleAndCount)(newTitle);
          newTitle = `${baseTitle} (${baseCount + 1})`;
          await checkForDuplicateDashboardTitle({
            title: newTitle,
            lastSavedTitle: currentState.title,
            copyOnSave: true,
            isTitleDuplicateConfirmed: false,
            onTitleDuplicate(speculativeSuggestion) {
              newTitle = speculativeSuggestion;
            }
          });
          switch (interactionMode) {
            case _public.ViewMode.EDIT:
              {
                customModalTitle = _i18n.i18n.translate('dashboard.topNav.editModeInteractiveSave.modalTitle', {
                  defaultMessage: 'Save as new dashboard'
                });
                break;
              }
            case _public.ViewMode.VIEW:
              {
                customModalTitle = _i18n.i18n.translate('dashboard.topNav.viewModeInteractiveSave.modalTitle', {
                  defaultMessage: 'Duplicate dashboard'
                });
                break;
              }
            default:
              {
                customModalTitle = undefined;
              }
          }
        }
        const dashboardDuplicateModal = /*#__PURE__*/_react.default.createElement(_save_modal.DashboardSaveModal, {
          tags: currentState.tags,
          title: newTitle,
          onClose: () => resolve(undefined),
          timeRestore: currentState.timeRestore,
          showStoreTimeOnSave: !lastSavedId,
          description: (_currentState$descrip = currentState.description) !== null && _currentState$descrip !== void 0 ? _currentState$descrip : '',
          showCopyOnSave: false,
          onSave: onSaveAttempt,
          customModalTitle: customModalTitle
        });
        this.clearOverlays();
        (0, _public2.showSaveModal)(dashboardDuplicateModal);
      } catch (error) {
        reject(error);
      }
    })();
  });
}