"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PluginsStatusService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _util = require("util");
var _coreStatusCommon = require("@kbn/core-status-common");
var _get_summary_status = require("./get_summary_status");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const STATUS_TIMEOUT_MS = 30 * 1000; // 30 seconds

const defaultStatus = {
  level: _coreStatusCommon.ServiceStatusLevels.unavailable,
  summary: 'Status not yet available'
};
class PluginsStatusService {
  constructor(deps, statusTimeoutMs = STATUS_TIMEOUT_MS) {
    (0, _defineProperty2.default)(this, "coreStatus", {
      elasticsearch: {
        ...defaultStatus
      },
      savedObjects: {
        ...defaultStatus
      }
    });
    (0, _defineProperty2.default)(this, "pluginData", void 0);
    (0, _defineProperty2.default)(this, "rootPlugins", void 0);
    // root plugins are those that do not have any dependencies
    (0, _defineProperty2.default)(this, "orderedPluginNames", void 0);
    (0, _defineProperty2.default)(this, "start$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "pluginData$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "pluginStatus", {});
    (0, _defineProperty2.default)(this, "pluginStatus$", new _rxjs.BehaviorSubject(this.pluginStatus));
    (0, _defineProperty2.default)(this, "reportedStatusSubscriptions", {});
    (0, _defineProperty2.default)(this, "reportingStatus", {});
    (0, _defineProperty2.default)(this, "newRegistrationsAllowed", true);
    (0, _defineProperty2.default)(this, "coreSubscription", void 0);
    this.statusTimeoutMs = statusTimeoutMs;
    this.pluginData = this.initPluginData(deps.pluginDependencies);
    this.rootPlugins = this.getRootPlugins();
    // plugin dependencies keys are already sorted
    this.orderedPluginNames = [...deps.pluginDependencies.keys()];
    this.coreSubscription = deps.core$.pipe((0, _rxjs.debounceTime)(10), (0, _rxjs.tap)(coreStatus => this.coreStatus = coreStatus), (0, _rxjs.map)(serviceStatuses => (0, _get_summary_status.getSummaryStatus)({
      serviceStatuses
    })),
    // no need to recalculate plugins statuses if core status hasn't changed
    (0, _rxjs.distinctUntilChanged)((previous, current) => previous.level === current.level)).subscribe(derivedCoreStatus => {
      this.updateRootPluginsStatuses(derivedCoreStatus);
      this.updateDependantStatuses(this.rootPlugins);
      this.emitCurrentStatus();
    });
  }

  /**
   * Register a status Observable for a specific plugin
   * @param {PluginName} plugin The name of the plugin
   * @param {Observable<ServiceStatus>} status$ An external Observable that must be trusted as the source of truth for the status of the plugin
   * @throws An error if the status registrations are not allowed
   */
  set(plugin, status$) {
    var _this$reportedStatusS;
    if (!this.newRegistrationsAllowed) {
      throw new Error(`Custom statuses cannot be registered after setup, plugin [${plugin}] attempted`);
    }
    this.reportingStatus[plugin] = true;
    // unsubscribe from any previous subscriptions. Ideally plugins should register a status Observable only once
    (_this$reportedStatusS = this.reportedStatusSubscriptions[plugin]) === null || _this$reportedStatusS === void 0 ? void 0 : _this$reportedStatusS.unsubscribe();

    // delete any derived statuses calculated before the custom status Observable was registered
    delete this.pluginStatus[plugin];
    const firstEmissionTimeout$ = this.start$.pipe((0, _rxjs.delay)(this.statusTimeoutMs), (0, _rxjs.map)(() => ({
      level: _coreStatusCommon.ServiceStatusLevels.unavailable,
      summary: `Status check timed out after ${this.statusTimeoutMs < 1000 ? `${this.statusTimeoutMs}ms` : `${this.statusTimeoutMs / 1000}s`}`
    })), (0, _rxjs.takeUntil)(status$));
    this.reportedStatusSubscriptions[plugin] = (0, _rxjs.merge)(firstEmissionTimeout$, status$).pipe((0, _rxjs.distinctUntilChanged)()).subscribe(status => {
      const {
        levelChanged,
        summaryChanged
      } = this.updatePluginReportedStatus(plugin, status);
      if (levelChanged) {
        this.updateDependantStatuses([plugin]);
      }
      if (levelChanged || summaryChanged) {
        this.emitCurrentStatus();
      }
    });
  }
  start() {
    // Prevent plugins from registering status Observables
    this.newRegistrationsAllowed = false;
    this.start$.next();
    this.start$.complete();
  }

  /**
   * Obtain an Observable of the status of all the plugins
   * @returns {Observable<Record<PluginName, PluginStatus>>} An Observable that will yield the current status of all plugins
   */
  getAll$() {
    return this.pluginStatus$.asObservable().pipe(
    // do not emit until we have a status for all plugins
    (0, _rxjs.filter)(all => Object.keys(all).length === this.orderedPluginNames.length), (0, _rxjs.distinctUntilChanged)(_util.isDeepStrictEqual));
  }

  /**
   * Obtain an Observable of the status of the dependencies of the given plugin
   * @param {PluginName} plugin the name of the plugin whose dependencies' status must be retreived
   * @returns {Observable<Record<PluginName, PluginStatus>>} An Observable that will yield the current status of the plugin's dependencies
   */
  getDependenciesStatus$(plugin) {
    const directDependencies = this.pluginData[plugin].dependencies;
    return this.getAll$().pipe((0, _rxjs.map)(allStatus => {
      const dependenciesStatus = {};
      directDependencies.forEach(dep => dependenciesStatus[dep] = allStatus[dep]);
      return dependenciesStatus;
    }), (0, _rxjs.debounceTime)(10));
  }

  /**
   * Obtain an Observable of the derived status of the given plugin
   * @param {PluginName} plugin the name of the plugin whose derived status must be retrieved
   * @returns {Observable<PluginStatus>} An Observable that will yield the derived status of the plugin
   */
  getDerivedStatus$(plugin) {
    return this.pluginData$.asObservable().pipe((0, _rxjs.map)(pluginData => {
      var _pluginData$plugin;
      return (_pluginData$plugin = pluginData[plugin]) === null || _pluginData$plugin === void 0 ? void 0 : _pluginData$plugin.derivedStatus;
    }), (0, _rxjs.filter)(status => !!status), (0, _rxjs.distinctUntilChanged)(_util.isDeepStrictEqual));
  }

  /**
   * Hook to be called at the stop lifecycle event
   */
  stop() {
    // Cancel all active subscriptions
    this.coreSubscription.unsubscribe();
    Object.values(this.reportedStatusSubscriptions).forEach(subscription => {
      subscription.unsubscribe();
    });
  }

  /**
   * Initialize a convenience data structure
   * that maintain up-to-date information about the plugins and their statuses
   * @param {ReadonlyMap<PluginName, PluginName[]>} pluginDependencies Information about the different plugins and their dependencies
   * @returns {PluginData}
   */
  initPluginData(pluginDependencies) {
    const pluginData = {};
    pluginDependencies.forEach((dependencies, name) => {
      pluginData[name] = {
        name,
        dependencies,
        reverseDependencies: [],
        derivedStatus: defaultStatus
      };
    });
    pluginDependencies.forEach((dependencies, name) => {
      dependencies.forEach(dependency => {
        pluginData[dependency].reverseDependencies.push(name);
      });
    });
    return pluginData;
  }

  /**
   * Create a list with all the root plugins.
   * Root plugins are all those plugins that do not have any dependency.
   * @returns {PluginName[]} a list with all the root plugins present in the provided deps
   */
  getRootPlugins() {
    return Object.keys(this.pluginData).filter(plugin => this.pluginData[plugin].dependencies.length === 0);
  }

  /**
   * Updates the root plugins statuses according to the current core services status
   */
  updateRootPluginsStatuses(derivedCoreStatus) {
    // note that the derived status is the same for all root plugins
    this.rootPlugins.forEach(pluginName => {
      this.pluginData[pluginName].derivedStatus = derivedCoreStatus;
      if (!this.reportingStatus[pluginName]) {
        // this root plugin has NOT registered any status Observable. Thus, its status is derived from core
        this.pluginStatus[pluginName] = derivedCoreStatus;
      }
    });
  }

  /**
   * Update the derived statuses of the specified plugins' dependant plugins,
   * If impacted plugins have not registered a custom status Observable, update their "current" status as well.
   * @param {PluginName[]} plugins The names of the plugins whose dependant plugins must be updated
   */
  updateDependantStatuses(plugins) {
    const toCheck = new Set();
    plugins.forEach(plugin => this.pluginData[plugin].reverseDependencies.forEach(revDep => toCheck.add(revDep)));

    // Note that we are updating the plugins in an ordered fashion.
    // This way, when updating plugin X (at depth = N),
    // all of its dependencies (at depth < N) have already been updated
    for (let i = 0; i < this.orderedPluginNames.length; ++i) {
      const current = this.orderedPluginNames[i];
      if (toCheck.has(current)) {
        // update the current plugin status
        this.updatePluginsStatus(current);
        // flag all its reverse dependencies to be checked
        // TODO flag them only IF the status of this plugin has changed, seems to break some tests
        this.pluginData[current].reverseDependencies.forEach(revDep => toCheck.add(revDep));
      }
    }
  }

  /**
   * Determine the derived status of the specified plugin and update it on the pluginData structure
   * Optionally, if the plugin has not registered a custom status Observable, update its "current" status as well
   * @param {PluginName} plugin The name of the plugin to be updated
   */
  updatePluginsStatus(plugin) {
    const newStatus = this.determineDerivedStatus(plugin);
    this.pluginData[plugin].derivedStatus = newStatus;
    if (!this.reportingStatus[plugin]) {
      // this plugin has NOT registered any status Observable.
      // Thus, its status is derived from its dependencies + core
      this.pluginStatus[plugin] = newStatus;
    }
  }

  /**
   * Determine the plugin's derived status (taking into account dependencies and core services)
   * @param {PluginName} pluginName the name of the plugin whose status must be determined
   * @returns {PluginStatus} The status of the plugin
   */
  determineDerivedStatus(pluginName) {
    if (Object.keys(this.reportingStatus).length) {
      // if at least one plugin has registered a status Observable... take into account plugin dependencies
      const pluginData = this.pluginData[pluginName];
      const dependenciesStatuses = Object.fromEntries(pluginData.dependencies.map(dependency => {
        var _this$pluginData$depe;
        return [dependency, (_this$pluginData$depe = this.pluginData[dependency].reportedStatus) !== null && _this$pluginData$depe !== void 0 ? _this$pluginData$depe : this.pluginData[dependency].derivedStatus];
      }));
      return (0, _get_summary_status.getSummaryStatus)({
        serviceStatuses: this.coreStatus,
        pluginStatuses: dependenciesStatuses
      });
    } else {
      // no plugins have registered a status Observable... infer status from Core services only
      return (0, _get_summary_status.getSummaryStatus)({
        serviceStatuses: this.coreStatus
      });
    }
  }

  /**
   * Updates the reported status for the given plugin.
   * @param {PluginName} pluginName The name of the plugin whose reported status must be updated
   * @param {ServiceStatus} status The newly reported status for that plugin
   * @return {Object} indicating whether the level and/or the summary have changed
   */
  updatePluginReportedStatus(pluginName, status) {
    const previousReportedStatus = this.pluginData[pluginName].reportedStatus;
    const reportedStatus = {
      ...status,
      reported: true
    };
    this.pluginData[pluginName].reportedStatus = reportedStatus;
    this.pluginStatus[pluginName] = reportedStatus;
    return {
      levelChanged: (previousReportedStatus === null || previousReportedStatus === void 0 ? void 0 : previousReportedStatus.level) !== reportedStatus.level,
      summaryChanged: (previousReportedStatus === null || previousReportedStatus === void 0 ? void 0 : previousReportedStatus.summary) !== reportedStatus.summary
    };
  }

  /**
   * Emit the current status to internal Subjects, effectively propagating it to observers.
   */
  emitCurrentStatus() {
    this.pluginData$.next(this.pluginData);
    // we must clone the plugin status to prevent future modifications from updating current emission
    this.pluginStatus$.next({
      ...this.pluginStatus
    });
  }
}
exports.PluginsStatusService = PluginsStatusService;