"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.arrayMapsAreEqual = arrayMapsAreEqual;
exports.setMapsAreEqual = setMapsAreEqual;
exports.setsAreEqual = setsAreEqual;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Determines if a given Set is equal to another given Set. Set types must be the same, and comparable.
 *
 * @param setA The first Set to compare
 * @param setB The second Set to compare
 * @returns {boolean} True if Set A is equal to Set B
 */
function setsAreEqual(setA, setB) {
  if (setA.size !== setB.size) return false;
  for (const element of setA) {
    if (!setB.has(element)) {
      return false;
    }
  }
  return true;
}

/**
 * Determines if a given map of arrays is equal to another given map of arrays.
 * Used for comparing namespace maps in saved object repo/security extension tests.
 *
 * @param mapA The first map to compare
 * @param mapB The second map to compare
 * @returns {boolean} True if map A is equal to map B
 */
function arrayMapsAreEqual(mapA, mapB) {
  if ((mapA === null || mapA === void 0 ? void 0 : mapA.size) !== (mapB === null || mapB === void 0 ? void 0 : mapB.size)) return false;
  for (const [key, valueA] of mapA) {
    const valueB = mapB === null || mapB === void 0 ? void 0 : mapB.get(key);
    if ((valueA === null || valueA === void 0 ? void 0 : valueA.length) !== (valueB === null || valueB === void 0 ? void 0 : valueB.length)) return false;
    if (!(0, _lodash.isEqual)(valueA === null || valueA === void 0 ? void 0 : valueA.sort(), valueB === null || valueB === void 0 ? void 0 : valueB.sort())) return false;
  }
  return true;
}

/**
 * Determines if a given Map of Sets is equal to another given Map of Sets.
 * Used for comparing typeMaps and enforceMaps in saved object repo/security extension tests.
 *
 * @param mapA The first map to compare
 * @param mapB The second map to compare
 * @returns {boolean} True if map A is equal to map B
 */
function setMapsAreEqual(mapA, mapB) {
  if ((mapA === null || mapA === void 0 ? void 0 : mapA.size) !== (mapB === null || mapB === void 0 ? void 0 : mapB.size)) return false;
  for (const [key, valueA] of mapA) {
    const valueB = mapB === null || mapB === void 0 ? void 0 : mapB.get(key);
    if (!valueB || !setsAreEqual(valueA, valueB)) return false;
  }
  return true;
}