"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBulkResolveRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const registerBulkResolveRoute = (router, {
  config,
  coreUsageData,
  logger,
  access
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.post({
    path: '/_bulk_resolve',
    options: {
      access,
      summary: `Resolve saved objects`,
      deprecated: true,
      description: `Retrieve multiple Kibana saved objects by ID, using any legacy URL aliases if they exist.
        Under certain circumstances, when Kibana is upgraded, saved object migrations may necessitate regenerating some object IDs to enable new features. When an object's ID is regenerated, a legacy URL alias is created for that object, preserving its old ID. In such a scenario, that object can be retrieved with the bulk resolve API using either its new ID or its old ID.`
    },
    validate: {
      body: _configSchema.schema.arrayOf(_configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }))
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'post',
      path: '/api/saved_objects/_bulk_resolve',
      request,
      logger
    });
    const types = [...new Set(request.body.map(({
      type
    }) => type))];
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsBulkResolve({
      request,
      types
    }).catch(() => {});
    const {
      savedObjects
    } = await context.core;
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfAnyTypeNotVisibleByAPI)(types, savedObjects.typeRegistry);
    }
    const result = await savedObjects.client.bulkResolve(request.body, {
      migrationVersionCompatibility: 'compatible'
    });
    return response.ok({
      body: result
    });
  }));
};
exports.registerBulkResolveRoute = registerBulkResolveRoute;