"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBulkGetRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const registerBulkGetRoute = (router, {
  config,
  coreUsageData,
  logger,
  access
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.post({
    path: '/_bulk_get',
    options: {
      access,
      summary: `Get saved objects`,
      deprecated: true
    },
    validate: {
      body: _configSchema.schema.arrayOf(_configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string(),
        fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        namespaces: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      }))
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'post',
      path: '/api/saved_objects/_bulk_get',
      request,
      logger
    });
    const types = [...new Set(request.body.map(({
      type
    }) => type))];
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsBulkGet({
      request,
      types
    }).catch(() => {});
    const {
      savedObjects
    } = await context.core;
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfAnyTypeNotVisibleByAPI)(types, savedObjects.typeRegistry);
    }
    const result = await savedObjects.client.bulkGet(request.body, {
      migrationVersionCompatibility: 'compatible'
    });
    return response.ok({
      body: result
    });
  }));
};
exports.registerBulkGetRoute = registerBulkGetRoute;