"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBulkCreateRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const registerBulkCreateRoute = (router, {
  config,
  coreUsageData,
  logger,
  access
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.post({
    path: '/_bulk_create',
    options: {
      access,
      summary: `Create saved objects`,
      deprecated: true
    },
    validate: {
      query: _configSchema.schema.object({
        overwrite: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      body: _configSchema.schema.arrayOf(_configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.maybe(_configSchema.schema.string()),
        attributes: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any()),
        version: _configSchema.schema.maybe(_configSchema.schema.string()),
        migrationVersion: _configSchema.schema.maybe(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.string())),
        coreMigrationVersion: _configSchema.schema.maybe(_configSchema.schema.string()),
        typeMigrationVersion: _configSchema.schema.maybe(_configSchema.schema.string()),
        references: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
          name: _configSchema.schema.string(),
          type: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))),
        initialNamespaces: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
          minSize: 1
        }))
      }))
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'post',
      path: '/api/saved_objects/_bulk_create',
      request,
      logger
    });
    const {
      overwrite
    } = request.query;
    const types = [...new Set(request.body.map(({
      type
    }) => type))];
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsBulkCreate({
      request,
      types
    }).catch(() => {});
    const {
      savedObjects
    } = await context.core;
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfAnyTypeNotVisibleByAPI)(types, savedObjects.typeRegistry);
    }
    const result = await savedObjects.client.bulkCreate(request.body, {
      overwrite,
      migrationVersionCompatibility: 'compatible'
    });
    return response.ok({
      body: result
    });
  }));
};
exports.registerBulkCreateRoute = registerBulkCreateRoute;