"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.outdatedDocumentsSearchTransform = void 0;
var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));
var _helpers = require("../../../model/helpers");
var _progress = require("../../../model/progress");
var _extract_errors = require("../../../model/extract_errors");
var _create_batches = require("../../../model/create_batches");
var _actions = require("../../actions");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const outdatedDocumentsSearchTransform = (state, res, context) => {
  // Increment the processed documents, no matter what the results are.
  // Otherwise the progress might look off when there are errors.
  const progress = (0, _progress.incrementProcessedProgress)(state.progress, state.outdatedDocuments.length);
  const discardCorruptObjects = context.discardCorruptObjects;
  if (Either.isRight(res) || (0, _actions.isTypeof)(res.left, 'documents_transform_failed') && discardCorruptObjects) {
    // we might have some transformation errors, but user has chosen to discard them
    if (state.corruptDocumentIds.length === 0 && state.transformErrors.length === 0 || discardCorruptObjects) {
      const documents = Either.isRight(res) ? res.right.processedDocs : res.left.processedDocs;
      let corruptDocumentIds = state.corruptDocumentIds;
      let transformErrors = state.transformErrors;
      if (Either.isLeft(res)) {
        corruptDocumentIds = [...state.corruptDocumentIds, ...res.left.corruptDocumentIds];
        transformErrors = [...state.transformErrors, ...res.left.transformErrors];
      }
      const batches = (0, _create_batches.createBatches)({
        documents,
        corruptDocumentIds,
        transformErrors,
        maxBatchSizeBytes: context.migrationConfig.maxBatchSizeBytes.getValueInBytes()
      });
      if (Either.isRight(batches)) {
        return {
          ...state,
          controlState: 'OUTDATED_DOCUMENTS_SEARCH_BULK_INDEX',
          bulkOperationBatches: batches.right,
          currentBatch: 0,
          hasTransformedDocs: true,
          progress
        };
      } else {
        return {
          ...state,
          controlState: 'FATAL',
          reason: (0, _extract_errors.fatalReasonDocumentExceedsMaxBatchSizeBytes)({
            _id: batches.left.documentId,
            docSizeBytes: batches.left.docSizeBytes,
            maxBatchSizeBytes: batches.left.maxBatchSizeBytes
          })
        };
      }
    } else {
      // We have seen corrupt documents and/or transformation errors
      // skip indexing and go straight to reading and transforming more docs
      return {
        ...state,
        controlState: 'OUTDATED_DOCUMENTS_SEARCH_READ',
        progress
      };
    }
  } else {
    if ((0, _actions.isTypeof)(res.left, 'documents_transform_failed')) {
      // continue to build up any more transformation errors before failing the migration.
      return {
        ...state,
        controlState: 'OUTDATED_DOCUMENTS_SEARCH_READ',
        corruptDocumentIds: [...state.corruptDocumentIds, ...res.left.corruptDocumentIds],
        transformErrors: [...state.transformErrors, ...res.left.transformErrors],
        hasTransformedDocs: false,
        progress
      };
    } else {
      (0, _helpers.throwBadResponse)(state, res);
    }
  }
};
exports.outdatedDocumentsSearchTransform = outdatedDocumentsSearchTransform;