"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractDiscardedCorruptDocs = extractDiscardedCorruptDocs;
exports.extractDiscardedUnknownDocs = extractDiscardedUnknownDocs;
exports.extractTransformFailuresReason = extractTransformFailuresReason;
exports.extractUnknownDocFailureReason = extractUnknownDocFailureReason;
exports.fatalReasonDocumentExceedsMaxBatchSizeBytes = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Constructs migration failure message strings from corrupt document ids and document transformation errors
 */
function extractTransformFailuresReason(resolveMigrationFailuresUrl, corruptDocumentIds, transformErrors) {
  const corruptDocumentIdReason = corruptDocumentIds.length > 0 ? ` ${corruptDocumentIds.length} corrupt saved object documents were found: ${corruptDocumentIds.join(', ')}\n` : '';
  // we have both the saved object Id and the stack trace in each `transformErrors` item.
  const transformErrorsReason = transformErrors.length > 0 ? ` ${transformErrors.length} transformation errors were encountered:\n` + transformErrors.map(errObj => {
    var _errObj$err$stack;
    return `- ${errObj.rawId}: ${(_errObj$err$stack = errObj.err.stack) !== null && _errObj$err$stack !== void 0 ? _errObj$err$stack : errObj.err.message}\n`;
  }).join('') : '';
  return `Migrations failed. Reason:${corruptDocumentIdReason}${transformErrorsReason}\n` + `To allow migrations to proceed, please delete or fix these documents.\n` + `Note that you can configure Kibana to automatically discard corrupt documents and transform errors for this migration.\n` + `Please refer to ${resolveMigrationFailuresUrl} for more information.`;
}
function extractDiscardedUnknownDocs(unknownDocs) {
  return `Kibana has been configured to discard unknown documents for this migration.\n` + `Therefore, the following documents with unknown types will not be taken into account and they will not be available after the migration:\n` + unknownDocs.map(doc => `- "${doc.id}" (type: "${doc.type}")\n`).join('');
}
function extractUnknownDocFailureReason(resolveMigrationFailuresUrl, unknownDocs) {
  return `Migration failed because some documents were found which use unknown saved object types:\n` + unknownDocs.map(doc => `- "${doc.id}" (type: "${doc.type}")\n`).join('') + `\nTo proceed with the migration you can configure Kibana to discard unknown saved objects for this migration.\n` + `Please refer to ${resolveMigrationFailuresUrl} for more information.`;
}
function extractDiscardedCorruptDocs(corruptDocumentIds, transformErrors) {
  return `Kibana has been configured to discard corrupt documents and documents that cause transform errors for this migration.\n` + `Therefore, the following documents will not be taken into account and they will not be available after the migration:\n` + corruptDocumentIds.map(id => `- "${id}" (corrupt)\n`).join('') + transformErrors.map(error => `- "${error.rawId}" (${error.err.message})\n`).join('');
}

/**
 * Constructs migration failure message string for doc exceeds max batch size in bytes
 */
const fatalReasonDocumentExceedsMaxBatchSizeBytes = ({
  _id,
  docSizeBytes,
  maxBatchSizeBytes
}) => `The document with _id "${_id}" is ${docSizeBytes} bytes which exceeds the configured maximum batch size of ${maxBatchSizeBytes} bytes. To proceed, please increase the 'migrations.maxBatchSizeBytes' Kibana configuration option and ensure that the Elasticsearch 'http.max_content_length' configuration option is set to an equal or larger value.`;
exports.fatalReasonDocumentExceedsMaxBatchSizeBytes = fatalReasonDocumentExceedsMaxBatchSizeBytes;