"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentMigrator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _utils = require("./pipelines/utils");
var _build_active_migrations = require("./build_active_migrations");
var _pipelines = require("./pipelines");
var _utils2 = require("./utils");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * A concrete implementation of the {@link IDocumentMigrator} interface.
 */
class DocumentMigrator {
  /**
   * Creates an instance of DocumentMigrator.
   *
   * @param {DocumentMigratorOptions} options
   * @prop {string} kibanaVersion - The current version of Kibana
   * @prop {SavedObjectTypeRegistry} typeRegistry - The type registry to get type migrations from
   * @prop {string} convertVersion - The version of Kibana in which documents can be converted to multi-namespace types
   * @prop {Logger} log - The migration logger
   */
  constructor(options) {
    (0, _defineProperty2.default)(this, "options", void 0);
    (0, _defineProperty2.default)(this, "migrations", void 0);
    (0, _defineProperty2.default)(this, "transformDown", (doc, {
      targetTypeVersion
    }) => {
      if (!this.migrations) {
        throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
      }
      const pipeline = new _pipelines.DocumentDowngradePipeline({
        document: doc,
        targetTypeVersion,
        typeTransforms: this.migrations[doc.type],
        kibanaVersion: this.options.kibanaVersion
      });
      const {
        document
      } = pipeline.run();
      const additionalDocs = [];
      return {
        document,
        additionalDocs
      };
    });
    this.options = options;
    this.migrate = (...args) => this.constructor.prototype.migrate.apply(this, args);
    this.migrateAndConvert = (...args) => this.constructor.prototype.migrateAndConvert.apply(this, args);
  }

  /**
   * Gets the latest pending version of each type.
   * Some migration objects won't have a latest migration version (they only contain reference transforms that are applied from other types).
   */
  getMigrationVersion({
    includeDeferred = true,
    migrationType = 'type'
  } = {}) {
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    return Object.entries(this.migrations).reduce((acc, [type, {
      latestVersion,
      immediateVersion
    }]) => {
      const version = includeDeferred ? latestVersion : immediateVersion;
      const latestMigrationVersion = migrationType === 'core' ? version.core : (0, _utils.maxVersion)(version.migrate, version.convert);
      if (latestMigrationVersion) {
        acc[type] = latestMigrationVersion;
      }
      return acc;
    }, {});
  }

  /**
   * Prepares active migrations and document transformer function.
   */
  prepareMigrations() {
    const {
      typeRegistry,
      kibanaVersion,
      log,
      convertVersion
    } = this.options;
    this.migrations = (0, _build_active_migrations.buildActiveMigrations)({
      typeRegistry,
      kibanaVersion,
      log,
      convertVersion
    });
  }

  /**
   * Migrates a document to the latest version.
   */
  migrate(doc, {
    allowDowngrade = false,
    targetTypeVersion
  } = {}) {
    const {
      document
    } = this.transform(doc, {
      allowDowngrade,
      targetTypeVersion
    });
    return document;
  }

  /**
   * Migrates a document to the latest version and applies type conversions if applicable. Also returns any additional document(s) that may
   * have been created during the transformation process.
   */
  migrateAndConvert(doc) {
    const {
      document,
      additionalDocs
    } = this.transform(doc, {
      convertNamespaceTypes: true,
      allowDowngrade: false
    });
    return [document, ...additionalDocs];
  }

  /**
   * Returns true if the provided document has a higher version that the `targetTypeVersion`
   * (defaulting to the last known version), false otherwise.
   */
  isDowngradeRequired(doc, {
    targetTypeVersion
  } = {}) {
    var _typeMigrations$lates;
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    const typeMigrations = this.migrations[doc.type];
    return (0, _utils2.downgradeRequired)(doc, (_typeMigrations$lates = typeMigrations === null || typeMigrations === void 0 ? void 0 : typeMigrations.latestVersion) !== null && _typeMigrations$lates !== void 0 ? _typeMigrations$lates : {}, targetTypeVersion);
  }
  transform(doc, {
    convertNamespaceTypes = false,
    allowDowngrade = false,
    targetTypeVersion
  } = {}) {
    var _typeMigrations$lates2;
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    const typeMigrations = this.migrations[doc.type];
    if ((0, _utils2.downgradeRequired)(doc, (_typeMigrations$lates2 = typeMigrations === null || typeMigrations === void 0 ? void 0 : typeMigrations.latestVersion) !== null && _typeMigrations$lates2 !== void 0 ? _typeMigrations$lates2 : {}, targetTypeVersion)) {
      var _doc$typeMigrationVer, _doc$migrationVersion;
      const currentVersion = (_doc$typeMigrationVer = doc.typeMigrationVersion) !== null && _doc$typeMigrationVer !== void 0 ? _doc$typeMigrationVer : (_doc$migrationVersion = doc.migrationVersion) === null || _doc$migrationVersion === void 0 ? void 0 : _doc$migrationVersion[doc.type];
      const latestVersion = targetTypeVersion !== null && targetTypeVersion !== void 0 ? targetTypeVersion : this.migrations[doc.type].latestVersion[_types.TransformType.Migrate];
      if (!allowDowngrade) {
        throw _boom.default.badData(`Document "${doc.id}" belongs to a more recent version of Kibana [${currentVersion}] when the last known version is [${latestVersion}].`);
      }
      return this.transformDown(doc, {
        targetTypeVersion: latestVersion
      });
    } else {
      return this.transformUp(doc, {
        convertNamespaceTypes,
        targetTypeVersion
      });
    }
  }
  transformUp(doc, {
    convertNamespaceTypes,
    targetTypeVersion
  }) {
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    const pipeline = new _pipelines.DocumentUpgradePipeline({
      document: doc,
      targetTypeVersion,
      migrations: this.migrations,
      kibanaVersion: this.options.kibanaVersion,
      convertNamespaceTypes
    });
    const {
      document,
      additionalDocs
    } = pipeline.run();
    return {
      document,
      additionalDocs
    };
  }
}
exports.DocumentMigrator = DocumentMigrator;