"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createIndex = void 0;
var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));
var TaskEither = _interopRequireWildcard(require("fp-ts/lib/TaskEither"));
var _function = require("fp-ts/lib/function");
var _catch_retryable_es_client_errors = require("./catch_retryable_es_client_errors");
var _constants = require("./constants");
var _wait_for_index_status = require("./wait_for_index_status");
var _es_errors = require("./es_errors");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function aliasArrayToRecord(aliases) {
  const result = {};
  for (const alias of aliases) {
    result[alias] = {};
  }
  return result;
}

/** @internal */

/** @internal */

/**
 * Creates an index with the given mappings
 *
 * @remarks
 * This method adds some additional logic to the ES create index API:
 *  - it is idempotent, if it gets called multiple times subsequent calls will
 *    wait for the first create operation to complete (up to 60s)
 *  - the first call will wait up to 120s for the cluster state and all shards
 *    to be updated.
 */
const createIndex = ({
  client,
  indexName,
  mappings,
  esCapabilities,
  aliases = [],
  timeout = _constants.DEFAULT_TIMEOUT,
  waitForIndexStatusTimeout = _constants.DEFAULT_TIMEOUT
}) => {
  const createIndexTask = () => {
    const aliasesObject = aliasArrayToRecord(aliases);
    const indexSettings = {
      // settings not being supported on serverless ES
      ...(esCapabilities.serverless ? {} : {
        // ES rule of thumb: shards should be several GB to 10's of GB, so
        // Kibana is unlikely to cross that limit.
        number_of_shards: _constants.INDEX_NUMBER_OF_SHARDS,
        auto_expand_replicas: _constants.INDEX_AUTO_EXPAND_REPLICAS,
        // Set an explicit refresh interval so that we don't inherit the
        // value from incorrectly configured index templates (not required
        // after we adopt system indices)
        refresh_interval: '1s',
        // Bump priority so that recovery happens before newer indices
        priority: 10
      }),
      // Increase the fields limit beyond the default of 1000
      mapping: {
        total_fields: {
          limit: 1500
        }
      }
    };
    return client.indices.create({
      index: indexName,
      // wait up to timeout until the following shards are available before
      // creating the index: primary, replica (only on multi node clusters)
      wait_for_active_shards: _constants.WAIT_FOR_ALL_SHARDS_TO_BE_ACTIVE,
      // Timeout for the cluster state to update and all shards to become
      // available. If the request doesn't complete within timeout,
      // acknowledged or shards_acknowledged would be false.
      timeout,
      mappings,
      aliases: aliasesObject,
      settings: {
        index: indexSettings
      }
    }).then(() => {
      return Either.right('create_index_succeeded');
    }).catch(error => {
      var _error$body, _error$body$error, _error$body2;
      if ((error === null || error === void 0 ? void 0 : (_error$body = error.body) === null || _error$body === void 0 ? void 0 : (_error$body$error = _error$body.error) === null || _error$body$error === void 0 ? void 0 : _error$body$error.type) === 'resource_already_exists_exception') {
        return Either.right('index_already_exists');
      } else if ((0, _es_errors.isClusterShardLimitExceeded)(error === null || error === void 0 ? void 0 : (_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.error)) {
        return Either.left({
          type: 'cluster_shard_limit_exceeded'
        });
      } else {
        throw error;
      }
    }).catch(_catch_retryable_es_client_errors.catchRetryableEsClientErrors);
  };
  return (0, _function.pipe)(createIndexTask, TaskEither.chain(res => {
    // Systematicaly wait until the target index has a 'green' status meaning
    // the primary (and on multi node clusters) the replica has been started
    // When the index status is 'green' we know that all shards were started
    // see https://github.com/elastic/kibana/issues/157968
    return (0, _function.pipe)((0, _wait_for_index_status.waitForIndexStatus)({
      client,
      index: indexName,
      timeout: waitForIndexStatusTimeout,
      status: 'green'
    }), TaskEither.map(() => res));
  }));
};
exports.createIndex = createIndex;