"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsExporter = void 0;
var _utils = require("@kbn/utils");
var _sort_objects = require("./sort_objects");
var _errors = require("./errors");
var _collect_exported_objects = require("./collect_exported_objects");
var _utils2 = require("./utils");
var _constants = require("./constants");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
/**
 * @internal
 */
var _savedObjectsClient = /*#__PURE__*/new WeakMap();
var _exportSizeLimit = /*#__PURE__*/new WeakMap();
var _typeRegistry = /*#__PURE__*/new WeakMap();
var _log = /*#__PURE__*/new WeakMap();
var _exportableTypes = /*#__PURE__*/new WeakMap();
class SavedObjectsExporter {
  constructor({
    savedObjectsClient,
    typeRegistry,
    exportSizeLimit,
    logger
  }) {
    _classPrivateFieldInitSpec(this, _savedObjectsClient, void 0);
    _classPrivateFieldInitSpec(this, _exportSizeLimit, void 0);
    _classPrivateFieldInitSpec(this, _typeRegistry, void 0);
    _classPrivateFieldInitSpec(this, _log, void 0);
    _classPrivateFieldInitSpec(this, _exportableTypes, void 0);
    _classPrivateFieldSet(_log, this, logger);
    _classPrivateFieldSet(_savedObjectsClient, this, savedObjectsClient);
    _classPrivateFieldSet(_exportSizeLimit, this, exportSizeLimit);
    _classPrivateFieldSet(_typeRegistry, this, typeRegistry);
    _classPrivateFieldSet(_exportableTypes, this, _classPrivateFieldGet(_typeRegistry, this).getImportableAndExportableTypes().map(type => type.name));
  }
  async exportByTypes(options) {
    _classPrivateFieldGet(_log, this).debug(`Initiating export for types: [${options.types}]`);
    const objects = await this.fetchByTypes(options);
    return this.processObjects(objects, _utils2.byIdAscComparator, {
      request: options.request,
      includeReferencesDeep: options.includeReferencesDeep,
      includeNamespaces: options.includeNamespaces,
      excludeExportDetails: options.excludeExportDetails,
      namespace: options.namespace
    });
  }
  async exportByObjects(options) {
    _classPrivateFieldGet(_log, this).debug(`Initiating export of [${options.objects.length}] objects.`);
    if (options.objects.length > _classPrivateFieldGet(_exportSizeLimit, this)) {
      throw _errors.SavedObjectsExportError.exportSizeExceeded(_classPrivateFieldGet(_exportSizeLimit, this));
    }
    const objects = await this.fetchByObjects(options);
    const comparator = (0, _utils2.getPreservedOrderComparator)(objects);
    return this.processObjects(objects, comparator, {
      request: options.request,
      includeReferencesDeep: options.includeReferencesDeep,
      includeNamespaces: options.includeNamespaces,
      excludeExportDetails: options.excludeExportDetails,
      namespace: options.namespace
    });
  }
  async processObjects(savedObjects, sortFunction, {
    request,
    excludeExportDetails = false,
    includeReferencesDeep = false,
    includeNamespaces = false,
    namespace
  }) {
    _classPrivateFieldGet(_log, this).debug(`Processing [${savedObjects.length}] saved objects.`);
    const {
      objects: collectedObjects,
      missingRefs: missingReferences,
      excludedObjects
    } = await (0, _collect_exported_objects.collectExportedObjects)({
      objects: savedObjects,
      includeReferences: includeReferencesDeep,
      namespace,
      request,
      typeRegistry: _classPrivateFieldGet(_typeRegistry, this),
      savedObjectsClient: _classPrivateFieldGet(_savedObjectsClient, this),
      logger: _classPrivateFieldGet(_log, this)
    });

    // sort with the provided sort function then with the default export sorting
    const exportedObjects = (0, _sort_objects.sortObjects)(collectedObjects.sort(sortFunction));

    // redact attributes that should not be exported
    const redactedObjects = includeNamespaces ? exportedObjects : exportedObjects.map(({
      namespaces,
      ...object
    }) => object);
    const exportDetails = {
      exportedCount: exportedObjects.length,
      missingRefCount: missingReferences.length,
      missingReferences,
      excludedObjectsCount: excludedObjects.length,
      excludedObjects
    };
    _classPrivateFieldGet(_log, this).debug(`Exporting [${redactedObjects.length}] saved objects.`);
    return (0, _utils.createListStream)([...redactedObjects, ...(excludeExportDetails ? [] : [exportDetails])]);
  }
  async fetchByObjects({
    objects,
    namespace
  }) {
    const bulkGetResult = await _classPrivateFieldGet(_savedObjectsClient, this).bulkGet(objects, {
      namespace
    });
    const erroredObjects = bulkGetResult.saved_objects.filter(obj => !!obj.error);
    if (erroredObjects.length) {
      throw _errors.SavedObjectsExportError.objectFetchError(erroredObjects);
    }
    return bulkGetResult.saved_objects;
  }
  async fetchByTypes({
    types,
    namespace,
    hasReference,
    search
  }) {
    if (types.includes(_constants.EXPORT_ALL_TYPES_TOKEN)) {
      types = _classPrivateFieldGet(_exportableTypes, this);
    }
    const finder = _classPrivateFieldGet(_savedObjectsClient, this).createPointInTimeFinder({
      type: types,
      hasReference,
      hasReferenceOperator: hasReference ? 'OR' : undefined,
      search,
      namespaces: namespace ? [namespace] : undefined
    });
    const hits = [];
    for await (const result of finder.find()) {
      hits.push(...result.saved_objects);
      if (hits.length > _classPrivateFieldGet(_exportSizeLimit, this)) {
        await finder.close();
        throw _errors.SavedObjectsExportError.exportSizeExceeded(_classPrivateFieldGet(_exportSizeLimit, this));
      }
    }

    // sorts server-side by _id, since it's only available in fielddata
    return hits
    // exclude the find-specific `score` property from the exported objects
    .map(({
      score,
      ...obj
    }) => obj).sort(_utils2.byIdAscComparator);
  }
}
exports.SavedObjectsExporter = SavedObjectsExporter;