"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkCreate = void 0;
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const performBulkCreate = async ({
  objects,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  migrator,
  extensions = {}
}) => {
  const {
    common: commonHelper,
    validation: validationHelper,
    encryption: encryptionHelper,
    preflight: preflightHelper,
    serializer: serializerHelper,
    migration: migrationHelper,
    user: userHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const {
    migrationVersionCompatibility,
    overwrite = false,
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    managed: optionsManaged
  } = options;
  const time = (0, _utils.getCurrentTime)();
  const createdBy = userHelper.getCurrentUserProfileUid();
  const updatedBy = createdBy;
  let preflightCheckIndexCounter = 0;
  const expectedResults = objects.map(object => {
    const {
      type,
      id: requestId,
      initialNamespaces,
      version,
      managed
    } = object;
    let error;
    let id = ''; // Assign to make TS happy, the ID will be validated (or randomly generated if needed) during getValidId below
    const objectManaged = managed;
    if (!allowedTypes.includes(type)) {
      error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
    } else {
      try {
        id = commonHelper.getValidId(type, requestId, version, overwrite);
        validationHelper.validateInitialNamespaces(type, initialNamespaces);
        validationHelper.validateOriginId(type, object);
      } catch (e) {
        error = e;
      }
    }
    if (error) {
      return (0, _utils.left)({
        id: requestId,
        type,
        error: (0, _utils.errorContent)(error)
      });
    }
    const method = requestId && overwrite ? 'index' : 'create';
    const requiresNamespacesCheck = requestId && registry.isMultiNamespace(type);
    return (0, _utils.right)({
      method,
      object: {
        ...object,
        id,
        managed: (0, _utils.setManaged)({
          optionsManaged,
          objectManaged
        })
      },
      ...(requiresNamespacesCheck && {
        preflightCheckIndex: preflightCheckIndexCounter++
      })
    });
  });
  const validObjects = expectedResults.filter(_utils.isRight);
  if (validObjects.length === 0) {
    // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
    return {
      // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'unknown' below)
      saved_objects: expectedResults.map(({
        value
      }) => value)
    };
  }
  const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
  const preflightCheckObjects = validObjects.filter(({
    value
  }) => value.preflightCheckIndex !== undefined).map(({
    value
  }) => {
    const {
      type,
      id,
      initialNamespaces
    } = value.object;
    const namespaces = initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : [namespaceString];
    return {
      type,
      id,
      overwrite,
      namespaces
    };
  });
  const preflightCheckResponse = await preflightHelper.preflightCheckForCreate(preflightCheckObjects);
  const authObjects = validObjects.map(element => {
    var _preflightResult$exis, _preflightResult$exis2;
    const {
      object,
      preflightCheckIndex: index
    } = element.value;
    const preflightResult = index !== undefined ? preflightCheckResponse[index] : undefined;
    return {
      type: object.type,
      id: object.id,
      initialNamespaces: object.initialNamespaces,
      existingNamespaces: (_preflightResult$exis = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$exis2 = preflightResult.existingDocument) === null || _preflightResult$exis2 === void 0 ? void 0 : _preflightResult$exis2._source.namespaces) !== null && _preflightResult$exis !== void 0 ? _preflightResult$exis : []
    };
  });
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeBulkCreate({
    namespace,
    objects: authObjects
  }));
  let bulkRequestIndexCounter = 0;
  const bulkCreateParams = [];
  const expectedBulkResults = await Promise.all(expectedResults.map(async expectedBulkGetResult => {
    if ((0, _utils.isLeft)(expectedBulkGetResult)) {
      return expectedBulkGetResult;
    }
    let savedObjectNamespace;
    let savedObjectNamespaces;
    let existingOriginId;
    let versionProperties;
    const {
      preflightCheckIndex,
      object: {
        initialNamespaces,
        version,
        ...object
      },
      method
    } = expectedBulkGetResult.value;
    if (preflightCheckIndex !== undefined) {
      var _existingDocument$_so;
      const preflightResult = preflightCheckResponse[preflightCheckIndex];
      const {
        type,
        id,
        existingDocument,
        error
      } = preflightResult;
      if (error) {
        const {
          metadata
        } = error;
        return (0, _utils.left)({
          id,
          type,
          error: {
            ...(0, _utils.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id)),
            ...(metadata && {
              metadata
            })
          }
        });
      }
      savedObjectNamespaces = initialNamespaces || (0, _utils.getSavedObjectNamespaces)(namespace, existingDocument);
      versionProperties = (0, _utils.getExpectedVersionProperties)(version);
      existingOriginId = existingDocument === null || existingDocument === void 0 ? void 0 : (_existingDocument$_so = existingDocument._source) === null || _existingDocument$_so === void 0 ? void 0 : _existingDocument$_so.originId;
    } else {
      if (registry.isSingleNamespace(object.type)) {
        savedObjectNamespace = initialNamespaces ? (0, _utils.normalizeNamespace)(initialNamespaces[0]) : namespace;
      } else if (registry.isMultiNamespace(object.type)) {
        savedObjectNamespaces = initialNamespaces || (0, _utils.getSavedObjectNamespaces)(namespace);
      }
      versionProperties = (0, _utils.getExpectedVersionProperties)(version);
    }

    // 1. If the originId has been *explicitly set* in the options (defined or undefined), respect that.
    // 2. Otherwise, preserve the originId of the existing object that is being overwritten, if any.
    const originId = Object.keys(object).includes('originId') ? object.originId : existingOriginId;
    const migrated = migrationHelper.migrateInputDocument({
      id: object.id,
      type: object.type,
      attributes: await encryptionHelper.optionallyEncryptAttributes(object.type, object.id, savedObjectNamespace,
      // only used for multi-namespace object types
      object.attributes),
      migrationVersion: object.migrationVersion,
      coreMigrationVersion: object.coreMigrationVersion,
      typeMigrationVersion: object.typeMigrationVersion,
      ...(savedObjectNamespace && {
        namespace: savedObjectNamespace
      }),
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      managed: (0, _utils.setManaged)({
        optionsManaged,
        objectManaged: object.managed
      }),
      updated_at: time,
      created_at: time,
      ...(createdBy && {
        created_by: createdBy
      }),
      ...(updatedBy && {
        updated_by: updatedBy
      }),
      references: object.references || [],
      originId
    });

    /**
     * If a validation has been registered for this type, we run it against the migrated attributes.
     * This is an imperfect solution because malformed attributes could have already caused the
     * migration to fail, but it's the best we can do without devising a way to run validations
     * inside the migration algorithm itself.
     */
    try {
      validationHelper.validateObjectForCreate(object.type, migrated);
    } catch (error) {
      return (0, _utils.left)({
        id: object.id,
        type: object.type,
        error
      });
    }
    const expectedResult = {
      esRequestIndex: bulkRequestIndexCounter++,
      requestedId: object.id,
      rawMigratedDoc: serializer.savedObjectToRaw(migrated)
    };
    bulkCreateParams.push({
      [method]: {
        _id: expectedResult.rawMigratedDoc._id,
        _index: commonHelper.getIndexForType(object.type),
        ...(overwrite && versionProperties)
      }
    }, expectedResult.rawMigratedDoc._source);
    return (0, _utils.right)(expectedResult);
  }));
  const bulkResponse = bulkCreateParams.length ? await client.bulk({
    refresh,
    require_alias: true,
    body: bulkCreateParams
  }) : undefined;
  const result = {
    saved_objects: expectedBulkResults.map(expectedResult => {
      var _bulkResponse$items$e;
      if ((0, _utils.isLeft)(expectedResult)) {
        return expectedResult.value;
      }
      const {
        requestedId,
        rawMigratedDoc,
        esRequestIndex
      } = expectedResult.value;
      const rawResponse = Object.values((_bulkResponse$items$e = bulkResponse === null || bulkResponse === void 0 ? void 0 : bulkResponse.items[esRequestIndex]) !== null && _bulkResponse$items$e !== void 0 ? _bulkResponse$items$e : {})[0];
      const error = (0, _utils.getBulkOperationError)(rawMigratedDoc._source.type, requestedId, rawResponse);
      if (error) {
        return {
          type: rawMigratedDoc._source.type,
          id: requestedId,
          error
        };
      }

      // When method == 'index' the bulkResponse doesn't include the indexed
      // _source so we return rawMigratedDoc but have to spread the latest
      // _seq_no and _primary_term values from the rawResponse.
      return serializerHelper.rawToSavedObject({
        ...rawMigratedDoc,
        ...{
          _seq_no: rawResponse._seq_no,
          _primary_term: rawResponse._primary_term
        }
      }, {
        migrationVersionCompatibility
      });
    })
  };
  return encryptionHelper.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, objects);
};
exports.performBulkCreate = performBulkCreate;