"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LoggerAdapter = void 0;
var _global_context = require("./global_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */
class LoggerAdapter {
  constructor(logger, globalContext = {}) {
    this.logger = logger;
    this.globalContext = globalContext;
  }

  /**
   * The current logger can be updated "on the fly", e.g. when the log config
   * has changed.
   *
   * This is not intended for external use, only internally in Kibana
   *
   * @internal
   */
  updateLogger(logger) {
    this.logger = logger;
  }

  /**
   * The current record of {@link GlobalContext} that can be updated on the fly.
   * Any updates via this method will be applied to all subsequent log entries.
   *
   * This is not intended for external use, only internally in Kibana
   *
   * @internal
   */
  updateGlobalContext(context) {
    this.globalContext = context;
  }
  trace(message, meta) {
    this.logger.trace(message, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  debug(message, meta) {
    this.logger.debug(message, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  info(message, meta) {
    this.logger.info(message, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  warn(errorOrMessage, meta) {
    this.logger.warn(errorOrMessage, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  error(errorOrMessage, meta) {
    this.logger.error(errorOrMessage, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  fatal(errorOrMessage, meta) {
    this.logger.fatal(errorOrMessage, (0, _global_context.mergeGlobalContext)(this.globalContext, meta));
  }
  log(record) {
    this.logger.log({
      ...record,
      meta: (0, _global_context.mergeGlobalContext)(this.globalContext, record.meta)
    });
  }
  isLevelEnabled(level) {
    return this.logger.isLevelEnabled(level);
  }
  get(...contextParts) {
    return this.logger.get(...contextParts);
  }
}
exports.LoggerAdapter = LoggerAdapter;