"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createXsrfPostAuthHandler = exports.createVersionCheckPostAuthHandler = exports.createRestrictInternalRoutesPostAuthHandler = exports.createCustomHeadersPreResponseHandler = exports.createBuildNrMismatchLoggerPreResponseHandler = void 0;
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _coreHttpCommon = require("@kbn/core-http-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const VERSION_HEADER = 'kbn-version';
const XSRF_HEADER = 'kbn-xsrf';
const KIBANA_NAME_HEADER = 'kbn-name';
const createXsrfPostAuthHandler = config => {
  const {
    allowlist,
    disableProtection
  } = config.xsrf;
  return (request, response, toolkit) => {
    if (disableProtection || allowlist.includes(request.route.path) || request.route.options.xsrfRequired === false) {
      return toolkit.next();
    }
    const hasVersionHeader = (VERSION_HEADER in request.headers);
    const hasXsrfHeader = (XSRF_HEADER in request.headers);
    if (!(0, _coreHttpRouterServerInternal.isSafeMethod)(request.route.method) && !hasVersionHeader && !hasXsrfHeader) {
      return response.badRequest({
        body: `Request must contain a ${XSRF_HEADER} header.`
      });
    }
    return toolkit.next();
  };
};
exports.createXsrfPostAuthHandler = createXsrfPostAuthHandler;
const createRestrictInternalRoutesPostAuthHandler = config => {
  const isRestrictionEnabled = config.restrictInternalApis;
  return (request, response, toolkit) => {
    const isInternalRoute = request.route.options.access === 'internal';
    if (isRestrictionEnabled && isInternalRoute && !request.isInternalApiRequest) {
      // throw 400
      return response.badRequest({
        body: `uri [${request.url.pathname}] with method [${request.route.method}] exists but is not available with the current configuration`
      });
    }
    return toolkit.next();
  };
};
exports.createRestrictInternalRoutesPostAuthHandler = createRestrictInternalRoutesPostAuthHandler;
const createVersionCheckPostAuthHandler = kibanaVersion => {
  return (request, response, toolkit) => {
    const requestVersion = request.headers[VERSION_HEADER];
    if (requestVersion && requestVersion !== kibanaVersion) {
      return response.badRequest({
        body: {
          message: `Browser client is out of date, please refresh the page ` + `("${VERSION_HEADER}" header was "${requestVersion}" but should be "${kibanaVersion}")`,
          attributes: {
            expected: kibanaVersion,
            got: requestVersion
          }
        }
      });
    }
    return toolkit.next();
  };
};
exports.createVersionCheckPostAuthHandler = createVersionCheckPostAuthHandler;
const createCustomHeadersPreResponseHandler = config => {
  const {
    name: serverName,
    securityResponseHeaders,
    customResponseHeaders,
    csp: {
      header: cspHeader,
      reportOnlyHeader: cspReportOnlyHeader
    }
  } = config;
  const additionalHeaders = {
    ...securityResponseHeaders,
    ...customResponseHeaders,
    'Content-Security-Policy': cspHeader,
    'Content-Security-Policy-Report-Only': cspReportOnlyHeader,
    [KIBANA_NAME_HEADER]: serverName
  };
  return (request, response, toolkit) => {
    return toolkit.next({
      headers: {
        ...additionalHeaders
      }
    });
  };
};
exports.createCustomHeadersPreResponseHandler = createCustomHeadersPreResponseHandler;
const shouldLogBuildNumberMismatch = (serverBuild, request, response) => {
  if (response.statusCode >= 400 && request.headers[_coreHttpCommon.KIBANA_BUILD_NR_HEADER] !== serverBuild.string) {
    const clientBuildNumber = parseInt(String(request.headers[_coreHttpCommon.KIBANA_BUILD_NR_HEADER]), 10);
    if (!isNaN(clientBuildNumber)) {
      return {
        log: true,
        clientBuild: clientBuildNumber
      };
    }
  }
  return {
    log: false
  };
};

/**
 * This should remain part of the logger prefix so that we can notify/track
 * when we see this logged for observability purposes.
 */
const BUILD_NUMBER_MISMATCH_LOGGER_NAME = 'kbn-build-number-mismatch';
const createBuildNrMismatchLoggerPreResponseHandler = (serverBuildNumber, log) => {
  const serverBuild = {
    number: serverBuildNumber,
    string: String(serverBuildNumber)
  };
  log = log.get(BUILD_NUMBER_MISMATCH_LOGGER_NAME);
  return (request, response, toolkit) => {
    const result = shouldLogBuildNumberMismatch(serverBuild, request, response);
    if (result.log === true) {
      const clientCompAdjective = result.clientBuild > serverBuildNumber ? 'newer' : 'older';
      log.warn(`Client build (${result.clientBuild}) is ${clientCompAdjective} than this Kibana server build (${serverBuildNumber}). The [${response.statusCode}] error status in req id [${request.id}] may be due to client-server incompatibility!`);
    }
    return toolkit.next();
  };
};
exports.createBuildNrMismatchLoggerPreResponseHandler = createBuildNrMismatchLoggerPreResponseHandler;