"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HttpService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _std = require("@kbn/std");
var _routerToOpenapispec = require("@kbn/router-to-openapispec");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _csp = require("./csp");
var _http_config = require("./http_config");
var _http_server = require("./http_server");
var _https_redirect_server = require("./https_redirect_server");
var _register_lifecycle_handlers = require("./register_lifecycle_handlers");
var _external_url = require("./external_url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */
class HttpService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "prebootServer", void 0);
    (0, _defineProperty2.default)(this, "isPrebootServerStopped", false);
    (0, _defineProperty2.default)(this, "httpServer", void 0);
    (0, _defineProperty2.default)(this, "httpsRedirectServer", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "configSubscription", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "env", void 0);
    (0, _defineProperty2.default)(this, "internalPreboot", void 0);
    (0, _defineProperty2.default)(this, "internalSetup", void 0);
    (0, _defineProperty2.default)(this, "requestHandlerContext", void 0);
    this.coreContext = coreContext;
    const {
      logger,
      configService,
      env
    } = coreContext;
    this.env = env;
    this.log = logger.get('http');
    this.config$ = (0, _rxjs.combineLatest)([configService.atPath(_http_config.config.path, {
      ignoreUnchanged: false
    }), configService.atPath(_csp.cspConfig.path), configService.atPath(_external_url.externalUrlConfig.path)]).pipe((0, _rxjs.map)(([http, csp, externalUrl]) => new _http_config.HttpConfig(http, csp, externalUrl)));
    const shutdownTimeout$ = this.config$.pipe((0, _rxjs.map)(({
      shutdownTimeout
    }) => shutdownTimeout));
    this.prebootServer = new _http_server.HttpServer(coreContext, 'Preboot', shutdownTimeout$);
    this.httpServer = new _http_server.HttpServer(coreContext, 'Kibana', shutdownTimeout$);
    this.httpsRedirectServer = new _https_redirect_server.HttpsRedirectServer(logger.get('http', 'redirect', 'server'));
  }
  async preboot(deps) {
    this.log.debug('setting up preboot server');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    const prebootSetup = await this.prebootServer.setup({
      config$: this.config$
    });
    prebootSetup.server.route({
      path: '/{p*}',
      method: '*',
      handler: (req, responseToolkit) => {
        this.log.debug(`Kibana server is not ready yet ${req.method}:${req.url.href}.`);

        // If server is not ready yet, because plugins or core can perform
        // long running tasks (build assets, saved objects migrations etc.)
        // we should let client know that and ask to retry after 30 seconds.
        return responseToolkit.response('Kibana server is not ready yet').code(503).header('Retry-After', '30');
      }
    });
    (0, _register_lifecycle_handlers.registerCoreHandlers)(prebootSetup, config, this.env, this.log);
    if (this.shouldListen(config)) {
      this.log.debug('starting preboot server');
      await this.prebootServer.start();
    }
    const prebootServerRequestHandlerContext = deps.context.createContextContainer();
    this.internalPreboot = {
      externalUrl: new _external_url.ExternalUrlConfig(config.externalUrl),
      csp: prebootSetup.csp,
      staticAssets: prebootSetup.staticAssets,
      basePath: prebootSetup.basePath,
      registerStaticDir: prebootSetup.registerStaticDir.bind(prebootSetup),
      auth: prebootSetup.auth,
      server: prebootSetup.server,
      registerRouteHandlerContext: (pluginOpaqueId, contextName, provider) => prebootServerRequestHandlerContext.registerContext(pluginOpaqueId, contextName, provider),
      registerRoutes: (path, registerCallback) => {
        const router = new _coreHttpRouterServerInternal.Router(path, this.log, prebootServerRequestHandlerContext.createHandler.bind(null, this.coreContext.coreId), {
          isDev: this.env.mode.dev,
          versionedRouterOptions: getVersionedRouterOptions(config)
        });
        registerCallback(router);
        prebootSetup.registerRouterAfterListening(router);
      },
      getServerInfo: prebootSetup.getServerInfo
    };
    return this.internalPreboot;
  }
  async setup(deps) {
    this.requestHandlerContext = deps.context.createContextContainer();
    this.configSubscription = this.config$.subscribe(() => {
      if (this.httpServer.isListening()) {
        // If the server is already running we can't make any config changes
        // to it, so we warn and don't allow the config to pass through.
        this.log.warn('Received new HTTP config after server was started. Config will **not** be applied.');
      }
    });
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    const {
      registerRouter,
      ...serverContract
    } = await this.httpServer.setup({
      config$: this.config$,
      executionContext: deps.executionContext
    });
    (0, _register_lifecycle_handlers.registerCoreHandlers)(serverContract, config, this.env, this.log);
    this.internalSetup = {
      ...serverContract,
      externalUrl: new _external_url.ExternalUrlConfig(config.externalUrl),
      createRouter: (path, pluginId = this.coreContext.coreId) => {
        const enhanceHandler = this.requestHandlerContext.createHandler.bind(null, pluginId);
        const router = new _coreHttpRouterServerInternal.Router(path, this.log, enhanceHandler, {
          isDev: this.env.mode.dev,
          versionedRouterOptions: getVersionedRouterOptions(config),
          pluginId
        });
        registerRouter(router);
        return router;
      },
      registerRouteHandlerContext: (pluginOpaqueId, contextName, provider) => this.requestHandlerContext.registerContext(pluginOpaqueId, contextName, provider)
    };
    return this.internalSetup;
  }

  // this method exists because we need the start contract to create the `CoreStart` used to start
  // the `plugin` and `legacy` services.
  getStartContract() {
    return {
      ...(0, _std.pick)(this.internalSetup, ['auth', 'basePath', 'getServerInfo', 'staticAssets']),
      isListening: () => this.httpServer.isListening()
    };
  }
  async start() {
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    if (this.shouldListen(config)) {
      this.log.debug('stopping preboot server');
      await this.prebootServer.stop();
      this.isPrebootServerStopped = true;

      // If a redirect port is specified, we start an HTTP server at this port and
      // redirect all requests to the SSL port.
      if (config.ssl.enabled && config.ssl.redirectHttpFromPort !== undefined) {
        await this.httpsRedirectServer.start(config);
      }
      if (config.oas.enabled) {
        this.log.info('Registering experimental OAS API');
        this.registerOasApi(config);
      }
      await this.httpServer.start();
    }
    return this.getStartContract();
  }
  registerOasApi(config) {
    var _this$internalSetup, _this$internalSetup2, _basePath$publicBaseU;
    const basePath = (_this$internalSetup = this.internalSetup) === null || _this$internalSetup === void 0 ? void 0 : _this$internalSetup.basePath;
    const server = (_this$internalSetup2 = this.internalSetup) === null || _this$internalSetup2 === void 0 ? void 0 : _this$internalSetup2.server;
    if (!basePath || !server) {
      throw new Error('Cannot register OAS API before server setup is complete');
    }
    const baseUrl = (_basePath$publicBaseU = basePath.publicBaseUrl) !== null && _basePath$publicBaseU !== void 0 ? _basePath$publicBaseU : `http://localhost:${config.port}${basePath.serverBasePath}`;
    server.route({
      path: '/api/oas',
      method: 'GET',
      handler: async (req, h) => {
        var _req$query, _req$query2, _req$query4, _req$query6, _req$query7;
        const version = (_req$query = req.query) === null || _req$query === void 0 ? void 0 : _req$query.version;
        let pathStartsWith;
        if (typeof ((_req$query2 = req.query) === null || _req$query2 === void 0 ? void 0 : _req$query2.pathStartsWith) === 'string') {
          pathStartsWith = [req.query.pathStartsWith];
        } else {
          var _req$query3;
          pathStartsWith = (_req$query3 = req.query) === null || _req$query3 === void 0 ? void 0 : _req$query3.pathStartsWith;
        }
        let excludePathsMatching;
        if (typeof ((_req$query4 = req.query) === null || _req$query4 === void 0 ? void 0 : _req$query4.excludePathsMatching) === 'string') {
          excludePathsMatching = [req.query.excludePathsMatching];
        } else {
          var _req$query5;
          excludePathsMatching = (_req$query5 = req.query) === null || _req$query5 === void 0 ? void 0 : _req$query5.excludePathsMatching;
        }
        const pluginId = (_req$query6 = req.query) === null || _req$query6 === void 0 ? void 0 : _req$query6.pluginId;
        const access = (_req$query7 = req.query) === null || _req$query7 === void 0 ? void 0 : _req$query7.access;
        if (access && !['public', 'internal'].some(a => a === access)) {
          return h.response({
            message: 'Invalid access query parameter. Must be one of "public" or "internal".'
          }).code(400);
        }
        return await (0, _rxjs.firstValueFrom)((0, _rxjs.of)(1).pipe(HttpService.generateOasSemaphore.acquire(), (0, _rxjs.mergeMap)(async () => {
          try {
            // Potentially quite expensive
            const result = (0, _routerToOpenapispec.generateOpenApiDocument)(this.httpServer.getRouters({
              pluginId
            }), {
              baseUrl,
              title: 'Kibana HTTP APIs',
              version: '0.0.0',
              // TODO get a better version here
              filters: {
                pathStartsWith,
                excludePathsMatching,
                access,
                version
              }
            });
            return h.response(result);
          } catch (e) {
            this.log.error(e);
            return h.response({
              message: e.message
            }).code(500);
          }
        })));
      },
      options: {
        app: {
          access: 'public'
        },
        auth: false,
        cache: {
          privacy: 'public',
          otherwise: 'must-revalidate'
        }
      }
    });
  }

  /**
   * Indicates if http server is configured to start listening on a configured port.
   * (if `server.autoListen` is not explicitly set to `false`.)
   *
   * @internal
   * */
  shouldListen(config) {
    return config.autoListen;
  }
  async stop() {
    var _this$configSubscript;
    (_this$configSubscript = this.configSubscription) === null || _this$configSubscript === void 0 ? void 0 : _this$configSubscript.unsubscribe();
    this.configSubscription = undefined;
    if (!this.isPrebootServerStopped) {
      this.isPrebootServerStopped = false;
      await this.prebootServer.stop();
    }
    await this.httpServer.stop();
    await this.httpsRedirectServer.stop();
  }
}
exports.HttpService = HttpService;
(0, _defineProperty2.default)(HttpService, "generateOasSemaphore", new _std.Semaphore(1));
function getVersionedRouterOptions(config) {
  return {
    defaultHandlerResolutionStrategy: config.versioned.versionResolution,
    useVersionResolutionStrategyForInternalPaths: config.versioned.useVersionResolutionStrategyForInternalPaths
  };
}