"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isCustomValidation = isCustomValidation;
exports.prepareVersionedRouteValidation = prepareVersionedRouteValidation;
exports.unwrapVersionedResponseBodyValidation = unwrapVersionedResponseBodyValidation;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function isCustomValidation(v) {
  return 'custom' in v;
}

/**
 * Utility for unwrapping versioned router response validation to
 * {@link RouteValidationSpec}.
 *
 * @param validation - versioned response body validation
 * @internal
 */
function unwrapVersionedResponseBodyValidation(validation) {
  if (isCustomValidation(validation)) {
    return validation.custom;
  }
  return validation();
}
function prepareValidation(validation) {
  if (validation.response) {
    const {
      unsafe,
      ...responseValidations
    } = validation.response;
    const result = {};
    for (const [key, {
      body
    }] of Object.entries(responseValidations)) {
      result[key] = {
        body: isCustomValidation(body) ? body : (0, _lodash.once)(body)
      };
    }
    return {
      ...validation,
      response: {
        ...validation.response,
        ...result
      }
    };
  }
  return validation;
}

// Integration tested in ./core_versioned_route.test.ts
function prepareVersionedRouteValidation(options) {
  if (typeof options.validate === 'function') {
    const validate = options.validate;
    return {
      ...options,
      validate: (0, _lodash.once)(() => prepareValidation(validate()))
    };
  } else if (typeof options.validate === 'object' && options.validate !== null) {
    return {
      ...options,
      validate: prepareValidation(options.validate)
    };
  }
  return options;
}