"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.lifecycleResponseFactory = exports.kibanaResponseFactory = exports.fileResponseFactory = exports.KibanaResponse = void 0;
var _mime = _interopRequireDefault(require("mime"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * A response data object, expected to returned as a result of {@link RequestHandler} execution
 * @internal
 */
class KibanaResponse {
  constructor(status, payload, options = {}) {
    this.status = status;
    this.payload = payload;
    this.options = options;
  }
}
exports.KibanaResponse = KibanaResponse;
const successResponseFactory = {
  ok: (options = {}) => new KibanaResponse(200, options.body, options),
  created: (options = {}) => new KibanaResponse(201, options.body, options),
  accepted: (options = {}) => new KibanaResponse(202, options.body, options),
  noContent: (options = {}) => new KibanaResponse(204, undefined, options),
  multiStatus: (options = {}) => new KibanaResponse(207, options.body, options)
};
const redirectionResponseFactory = {
  redirected: options => new KibanaResponse(302, options.body, options)
};
const notModifiedResponseFactory = {
  notModified: (options = {}) => new KibanaResponse(304, undefined, options)
};
const errorResponseFactory = {
  badRequest: (options = {}) => new KibanaResponse(400, options.body || 'Bad Request', options),
  unauthorized: (options = {}) => new KibanaResponse(401, options.body || 'Unauthorized', options),
  forbidden: (options = {}) => new KibanaResponse(403, options.body || 'Forbidden', options),
  notFound: (options = {}) => new KibanaResponse(404, options.body || 'Not Found', options),
  conflict: (options = {}) => new KibanaResponse(409, options.body || 'Conflict', options),
  unprocessableContent: (options = {}) => new KibanaResponse(422, options.body || 'Unprocessable Content', options),
  customError: options => {
    if (!options || !options.statusCode) {
      throw new Error(`options.statusCode is expected to be set. given options: ${options && options.statusCode}`);
    }
    if (options.statusCode < 400 || options.statusCode >= 600) {
      throw new Error(`Unexpected Http status code. Expected from 400 to 599, but given: ${options.statusCode}`);
    }
    return new KibanaResponse(options.statusCode, options.body, options);
  }
};
const fileResponseFactory = exports.fileResponseFactory = {
  file: options => {
    var _ref;
    const {
      body,
      bypassErrorFormat,
      fileContentSize,
      headers,
      filename,
      fileContentType,
      bypassFileNameEncoding
    } = options;
    const reponseFilename = bypassFileNameEncoding ? filename : encodeURIComponent(filename);
    const responseBody = typeof body === 'string' ? Buffer.from(body) : body;
    if (!responseBody) {
      throw new Error(`options.body is expected to be set.`);
    }
    const responseContentType = (_ref = fileContentType !== null && fileContentType !== void 0 ? fileContentType : _mime.default.getType(filename)) !== null && _ref !== void 0 ? _ref : 'application/octet-stream';
    const responseContentLength = typeof fileContentSize === 'number' ? fileContentSize : Buffer.isBuffer(responseBody) ? responseBody.length : '';
    return new KibanaResponse(200, responseBody, {
      bypassErrorFormat,
      headers: {
        ...headers,
        'content-type': `${responseContentType}`,
        'content-length': `${responseContentLength}`,
        'content-disposition': `attachment; filename=${reponseFilename}`,
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options
        'x-content-type-options': 'nosniff'
      }
    });
  }
};
const kibanaResponseFactory = exports.kibanaResponseFactory = {
  ...successResponseFactory,
  ...redirectionResponseFactory,
  ...notModifiedResponseFactory,
  ...errorResponseFactory,
  ...fileResponseFactory,
  custom: options => {
    if (!options || !options.statusCode) {
      throw new Error(`options.statusCode is expected to be set. given options: ${options && options.statusCode}`);
    }
    const {
      statusCode: code,
      body,
      ...rest
    } = options;
    return new KibanaResponse(code, body, {
      ...rest
    });
  }
};
const lifecycleResponseFactory = exports.lifecycleResponseFactory = {
  ...redirectionResponseFactory,
  ...errorResponseFactory
};