"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsEventStreamClient = void 0;
var _esQuery = require("@kbn/es-query");
var _es_event_stream_names = require("./es_event_stream_names");
var _es_event_stream_initializer = require("./init/es_event_stream_initializer");
var _util = require("./util");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const sort = [{
  // By default we always sort by event timestamp descending.
  '@timestamp': {
    order: 'desc'
  },
  // Tie breakers for events with the same timestamp.
  subjectId: {
    order: 'desc'
  },
  objectId: {
    order: 'desc'
  },
  predicate: {
    order: 'desc'
  }
}];
var _names = /*#__PURE__*/new WeakMap();
class EsEventStreamClient {
  constructor(deps) {
    _classPrivateFieldInitSpec(this, _names, void 0);
    this.deps = deps;
    _classPrivateFieldSet(_names, this, new _es_event_stream_names.EsEventStreamNames(deps.baseName));
  }
  async initialize() {
    const initializer = new _es_event_stream_initializer.EsEventStreamInitializer({
      names: _classPrivateFieldGet(_names, this),
      kibanaVersion: this.deps.kibanaVersion,
      logger: this.deps.logger,
      esClient: this.deps.esClient
    });
    await initializer.initialize();
  }
  async writeEvents(events) {
    if (events.length === 0) return;
    const esClient = await this.deps.esClient;
    const operations = [];
    for (const event of events) {
      const dto = (0, _util.eventToDto)(event);
      operations.push({
        create: {}
      }, dto);
    }
    const {
      errors
    } = await esClient.bulk({
      index: _classPrivateFieldGet(_names, this).dataStream,
      operations
    }, {
      maxRetries: 0
    });
    if (errors) {
      throw new Error('Some events failed to be indexed.');
    }
  }
  async tail(limit = 100) {
    return (await this.filter({
      limit
    })).events;
  }
  async filter(options) {
    var _options$limit;
    const esClient = await this.deps.esClient;
    const topLevelNodes = [];
    if (options.subject && options.subject.length) {
      topLevelNodes.push(_esQuery.nodeBuilder.or(options.subject.map(([type, id]) => !id ? _esQuery.nodeBuilder.is('subjectType', type) : _esQuery.nodeBuilder.and([_esQuery.nodeBuilder.is('subjectType', type), _esQuery.nodeBuilder.is('subjectId', id)]))));
    }
    if (options.object && options.object.length) {
      topLevelNodes.push(_esQuery.nodeBuilder.or(options.object.map(([type, id]) => !id ? _esQuery.nodeBuilder.is('objectType', type) : _esQuery.nodeBuilder.and([_esQuery.nodeBuilder.is('objectType', type), _esQuery.nodeBuilder.is('objectId', id)]))));
    }
    if (options.predicate && options.predicate.length) {
      topLevelNodes.push(_esQuery.nodeBuilder.or(options.predicate.map(type => _esQuery.nodeBuilder.is('predicate', type))));
    }
    if (options.transaction && options.transaction.length) {
      topLevelNodes.push(_esQuery.nodeBuilder.or(options.transaction.map(id => _esQuery.nodeBuilder.is('txId', id))));
    }
    if (options.from) {
      const from = new Date(options.from).toISOString();
      const node = _esQuery.nodeBuilder.range('@timestamp', 'gte', from);
      topLevelNodes.push(node);
    }
    if (options.to) {
      const to = new Date(options.to).toISOString();
      const node = _esQuery.nodeBuilder.range('@timestamp', 'lte', to);
      topLevelNodes.push(node);
    }
    const query = (0, _esQuery.toElasticsearchQuery)(_esQuery.nodeBuilder.and(topLevelNodes));
    const size = (_options$limit = options.limit) !== null && _options$limit !== void 0 ? _options$limit : 100;
    const request = {
      index: _classPrivateFieldGet(_names, this).dataStream,
      query,
      sort,
      size,
      track_total_hits: false
    };
    if (options.cursor) {
      request.search_after = JSON.parse(options.cursor);
    }
    const res = await esClient.search(request);
    const events = res.hits.hits.map(hit => (0, _util.dtoToEvent)(hit._source));
    const lastHit = res.hits.hits[res.hits.hits.length - 1];
    let cursor = '';
    if (events.length >= size && lastHit && lastHit.sort) {
      cursor = JSON.stringify(lastHit.sort);
    }
    return {
      cursor,
      events
    };
  }
}
exports.EsEventStreamClient = EsEventStreamClient;